/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.module.open.serviceimpl;

import cn.hutool.core.date.DateUtil;
import cn.hutool.core.util.StrUtil;
import jakarta.annotation.Nullable;
import jakarta.annotation.Resource;
import jakarta.persistence.criteria.*;
import lombok.extern.slf4j.Slf4j;
import org.springframework.cache.annotation.CacheConfig;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import plus.hiver.common.base.HiverBaseServiceImpl;
import plus.hiver.common.redis.RedisTemplateHelper;
import plus.hiver.common.vo.SearchVo;
import plus.hiver.module.open.dao.ClientDao;
import plus.hiver.module.open.entity.Client;
import plus.hiver.module.open.service.ClientService;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * 客户端接口实现
 *
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
@Slf4j
@Service
@Transactional
@CacheConfig(cacheNames = "client")
public class ClientServiceImpl extends HiverBaseServiceImpl<Client, Long, ClientDao> implements ClientService {
    @Resource
    private RedisTemplateHelper redisTemplateHelper;

    protected ClientServiceImpl(ClientDao dao) {
        super(dao);
    }

    @Override
    @Cacheable(key = "#id")
    public Client get(Long id) {
        // 避免缓存穿透
        String result = redisTemplateHelper.get("client::" + id);
        if ("null".equals(result)) {
            return null;
        }
        Client client = dao.findById(id).orElse(null);
        if (client == null) {
            redisTemplateHelper.set("client::" + id, "null", 5L, TimeUnit.MINUTES);
        }
        return client;
    }

    @Override
    @CacheEvict(key = "#client.id")
    public Client update(Client client) {
        return dao.saveAndFlush(client);
    }

    @Override
    @CacheEvict(key = "#id")
    public void delete(Long id) {
        dao.deleteById(id);
    }

    @Override
    public Page<Client> findByCondition(Client client, SearchVo searchVo, Pageable pageable) {
        return dao.findAll(new Specification<Client>() {
            @Nullable
            @Override
            public Predicate toPredicate(Root<Client> root, CriteriaQuery<?> cq, CriteriaBuilder cb) {
                Path<String> nameField = root.get("name");
                Path<String> homeUriField = root.get("homeUri");
                Path<Date> createTimeField = root.get("createTime");

                List<Predicate> list = new ArrayList<>();

                //模糊搜素
                if (StrUtil.isNotBlank(client.getName())) {
                    list.add(cb.like(nameField, '%' + client.getName() + '%'));
                }
                if (StrUtil.isNotBlank(client.getHomeUri())) {
                    list.add(cb.like(homeUriField, '%' + client.getHomeUri() + '%'));
                }

                //创建时间
                if (StrUtil.isNotBlank(searchVo.getStartDate()) && StrUtil.isNotBlank(searchVo.getEndDate())) {
                    Date start = DateUtil.parse(searchVo.getStartDate());
                    Date end = DateUtil.parse(searchVo.getEndDate());
                    list.add(cb.between(createTimeField, start, DateUtil.endOfDay(end)));
                }

                Predicate[] arr = new Predicate[list.size()];
                cq.where(list.toArray(arr));
                return null;
            }
        }, pageable);
    }
}
