/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.module.quartz.controller;

import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.annotation.Resource;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.Page;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.bind.annotation.*;
import plus.hiver.common.annotation.PermissionTag;
import plus.hiver.common.api.Result;
import plus.hiver.common.constant.HiverConstant;
import plus.hiver.common.utils.PageUtil;
import plus.hiver.common.utils.ResultUtil;
import plus.hiver.common.utils.SnowFlakeUtil;
import plus.hiver.common.vo.PageVo;
import plus.hiver.module.quartz.entity.QuartzJob;
import plus.hiver.module.quartz.service.QuartzJobService;
import plus.hiver.module.quartz.service.QuartzLogService;
import plus.hiver.module.quartz.utils.SchedulerUtil;

/**
 * 定时任务管理接口
 *
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
@Slf4j
@RestController
@Tag(name = "定时任务执行日志管理接口")
@AllArgsConstructor
@RequestMapping("/hiver/quartzJob")
@PermissionTag(permission = "quartzJob:*" )
@Transactional
public class QuartzJobController {
    @Resource
    private QuartzJobService quartzJobService;

    @Resource
    private QuartzLogService quartzLogService;

    @Resource
    private SchedulerUtil schedulerUtil;

    @GetMapping(value = "/getAllByPage")
    @Operation(summary = "获取所有定时任务")
    public Result<Page<QuartzJob>> getAllByPage(String key, PageVo page) {
        Page<QuartzJob> data = quartzJobService.findByCondition(key, PageUtil.initPage(page));
        return new ResultUtil<Page<QuartzJob>>().setData(data);
    }

    @PostMapping(value = "/add")
    @Operation(summary = "添加定时任务")
    public Result addJob(@RequestBody QuartzJob job) {
        job.setId(SnowFlakeUtil.nextId());
        QuartzJob quartzJob = quartzJobService.findByJobClassName(job.getJobClassName());
        if (quartzJob != null) {
            return ResultUtil.error("该定时任务类名已存在");
        }
        schedulerUtil.add(job);
        quartzJobService.save(job);
        return ResultUtil.success("创建定时任务成功");
    }

    @PostMapping(value = "/edit")
    @Operation(summary = "更新定时任务")
    public Result editJob(@RequestBody QuartzJob job) {
        schedulerUtil.delete(job);
        schedulerUtil.add(job);
        job.setStatus(HiverConstant.STATUS_NORMAL);
        quartzJobService.update(job);
        return ResultUtil.success("更新定时任务成功");
    }

    @PostMapping(value = "/pause")
    @Operation(summary = "暂停定时任务")
    public Result pauseJob(QuartzJob job) {
        schedulerUtil.pauseJob(job);
        job.setStatus(HiverConstant.STATUS_DISABLE);
        quartzJobService.update(job);
        return ResultUtil.success("暂停定时任务成功");
    }

    @PostMapping(value = "/resume")
    @Operation(summary = "恢复定时任务")
    public Result resumeJob(QuartzJob job) {
        schedulerUtil.resumeJob(job);
        job.setStatus(HiverConstant.STATUS_NORMAL);
        quartzJobService.update(job);
        return ResultUtil.success("恢复定时任务成功");
    }

    @PostMapping(value = "/changeRecordLog")
    @Operation(summary = "改变记录日志开关")
    public Result changeRecordLog(Long id) {
        QuartzJob job = quartzJobService.get(id);
        job.setIsRecordLog(!job.getIsRecordLog());
        quartzJobService.update(job);
        return ResultUtil.success("操作成功");
    }

    @PostMapping(value = "/delByIds")
    @Operation(summary = "删除定时任务")
    public Result deleteJob(@RequestParam Long[] ids) {
        for (Long id : ids) {
            QuartzJob job = quartzJobService.get(id);
            schedulerUtil.delete(job);
            quartzJobService.delete(job);
            // 删除关联执行日志
            quartzLogService.deleteByJobClassName(job.getJobClassName());
        }
        return ResultUtil.success("删除定时任务成功");
    }
}
