/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.module.quartz.serviceimpl;

import cn.hutool.core.util.StrUtil;
import jakarta.persistence.criteria.*;
import jakarta.transaction.Transactional;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.lang.Nullable;
import org.springframework.stereotype.Service;
import plus.hiver.common.base.HiverBaseServiceImpl;
import plus.hiver.module.quartz.dao.QuartzJobDao;
import plus.hiver.module.quartz.entity.QuartzJob;
import plus.hiver.module.quartz.service.QuartzJobService;

import java.util.ArrayList;
import java.util.List;

/**
 * 定时任务接口实现
 *
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
@Slf4j
@Service
@Transactional
public class QuartzJobServiceImpl extends HiverBaseServiceImpl<QuartzJob, Long, QuartzJobDao> implements QuartzJobService {
    public QuartzJobServiceImpl(QuartzJobDao dao) {
        super(dao);
    }

    @Override
    public QuartzJob findByJobClassName(String jobClassName) {
        List<QuartzJob> list = dao.findByJobClassName(jobClassName);
        if (list == null || list.size() == 0) {
            return null;
        }
        return list.get(0);
    }

    @Override
    public Page<QuartzJob> findByCondition(String key, Pageable pageable) {
        return dao.findAll(new Specification<QuartzJob>() {
            @Nullable
            @Override
            public Predicate toPredicate(Root<QuartzJob> root, CriteriaQuery<?> cq, CriteriaBuilder cb) {
                Path<String> titleField = root.get("title");
                Path<String> jobClassNameField = root.get("jobClassName");
                Path<String> parameterField = root.get("parameter");

                List<Predicate> list = new ArrayList<>();

                // 模糊搜素
                if (StrUtil.isNotBlank(key)) {
                    Predicate p1 = cb.like(jobClassNameField, '%' + key + '%');
                    Predicate p2 = cb.like(parameterField, '%' + key + '%');
                    Predicate p3 = cb.like(titleField, '%' + key + '%');
                    list.add(cb.or(p1, p2, p3));
                }

                Predicate[] arr = new Predicate[list.size()];
                cq.where(list.toArray(arr));
                return null;
            }
        }, pageable);
    }
}
