/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.rustfs.controller;

import cn.hutool.core.util.StrUtil;
import com.google.gson.Gson;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.annotation.Resource;
import jakarta.servlet.http.HttpServletRequest;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.multipart.MultipartFile;
import plus.hiver.common.annotation.PermissionTag;
import plus.hiver.common.api.Result;
import plus.hiver.common.constant.SettingConstant;
import plus.hiver.common.entity.Setting;
import plus.hiver.common.service.SettingService;
import plus.hiver.common.utils.Base64DecodeMultipartFile;
import plus.hiver.common.utils.CommonUtil;
import plus.hiver.common.utils.IpInfoUtil;
import plus.hiver.common.utils.ResultUtil;
import plus.hiver.module.system.vo.OssSetting;
import plus.hiver.rustfs.entity.File;
import plus.hiver.rustfs.manage.FileManageFactory;
import plus.hiver.rustfs.service.FileService;

import java.io.InputStream;

/**
 * 文件上传接口
 *
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
@Slf4j
@RestController
@Tag(name = "文件上传接口")
@PermissionTag(permission = "upload:*" )
@RequestMapping("/hiver/upload")
public class UploadController {
    @Value("${hiver.maxUploadFile}")
    private Long maxUploadFile;

    @Resource
    private IpInfoUtil ipInfoUtil;

    @Resource
    private FileManageFactory fileManageFactory;

    @Resource
    private SettingService settingService;

    @Resource
    private FileService fileService;

    @PostMapping(value = "/file")
    @Operation(summary = "文件上传")
    public Result upload(@RequestParam(required = false) MultipartFile file,
                         @RequestParam(required = false) String base64,
                         @RequestParam(required = false) Long categoryId,
                         HttpServletRequest request) {
        if (file == null && StrUtil.isBlank(base64)) {
            return ResultUtil.error("缺少文件参数");
        }
        if (file != null && file.getSize() > maxUploadFile * 1024 * 1024) {
            return ResultUtil.error("文件大小过大，不能超过" + maxUploadFile + "MB");
        }
        Setting setting = settingService.findByKey(SettingConstant.OSS_USED);
        if (setting == null || StrUtil.isBlank(setting.getValue())) {
            return ResultUtil.error(501, "您还未配置OSS存储服务");
        }
        if (StrUtil.isNotBlank(base64)) {
            // base64上传
            file = Base64DecodeMultipartFile.base64Convert(base64);
        }
        String fKey = CommonUtil.renamePic(file.getOriginalFilename());
        if(setting.getValue().equals(SettingConstant.RUSTFS_OSS)) {
            OssSetting os = new Gson().fromJson(settingService.findByKey(SettingConstant.RUSTFS_OSS).getValue(), OssSetting.class);
            fKey = os.getBucket();
        }
        String result = "";
        File f = new File();
        try {
            InputStream inputStream = file.getInputStream();
            // 上传至第三方云服务或服务器
            result = fileManageFactory.getFileManage(null).inputStreamUpload(inputStream, fKey, file);
            // 保存数据信息至数据库
            f.setLocation(getType(setting.getValue())).setTitle(file.getOriginalFilename()).setSize(file.getSize())
                    .setType(file.getContentType()).setFKey(fKey).setUrl(result).setCategoryId(categoryId);
            fileService.save(f);
        } catch (Exception e) {
            log.error(e.toString());
            return ResultUtil.error(e.toString());
        }
        if (setting.getValue().equals(SettingConstant.LOCAL_OSS)) {
            OssSetting os = new Gson().fromJson(settingService.findByKey(SettingConstant.LOCAL_OSS).getValue(), OssSetting.class);
            result = os.getHttp() + os.getEndpoint() + "/" + f.getId();
        }
        return ResultUtil.data(result);
    }

    public Integer getType(String type) {
        switch (type) {
            case SettingConstant.RUSTFS_OSS:
                return 1;
            case SettingConstant.LOCAL_OSS:
                return 0;
            default:
                return -1;
        }
    }
}
