/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.rustfs.manage;

import cn.hutool.core.util.StrUtil;
import jakarta.annotation.Resource;
import org.springframework.stereotype.Component;
import plus.hiver.common.constant.SettingConstant;
import plus.hiver.common.entity.Setting;
import plus.hiver.common.exception.HiverException;
import plus.hiver.common.service.SettingService;
import plus.hiver.rustfs.manage.impl.LocalFileManage;
import plus.hiver.rustfs.manage.impl.RustfsManage;

import java.util.HashMap;
import java.util.Map;

/**
 * 工厂模式
 *
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
@Component
public class FileManageFactory {
    @Resource
    private SettingService settingService;

    @Resource
    private LocalFileManage localFileManage;

    @Resource
    private RustfsManage rustfsManage;

    private static volatile Map<String, FileManage> fileManages;

    /**
     * 单例初始化所有类型文件对应实现类
     *
     * @return 结果
     */
    private Map<String, FileManage> getFileManages() {
        if (fileManages == null) {
            synchronized (this) {
                if (fileManages == null) {
                    fileManages = new HashMap<>(16);
                    fileManages.put(SettingConstant.OSS_TYPE.LOCAL_OSS.name(), localFileManage);
                    fileManages.put(SettingConstant.OSS_TYPE.RUSTFS_OSS.name(), rustfsManage);
                }
            }
        }
        return fileManages;
    }

    /**
     * 管理文件时使用
     *
     * @return 结果
     */
    public FileManage getFileManage() {
        return getFileManage(null);
    }

    /**
     * 使用配置的服务上传时location传入null 管理文件时需传入存储位置location
     *
     * @param location 存储位置
     * @return 结果
     */
    public FileManage getFileManage(Integer location) {
        Setting setting = settingService.findByKey(SettingConstant.OSS_USED);
        if (setting == null || StrUtil.isBlank(setting.getValue())) {
            throw new HiverException("您还未配置OSS存储服务");
        }
        String type = setting.getValue();
        Map<String, FileManage> manages = getFileManages();
        for (Map.Entry<String, FileManage> entry : manages.entrySet()) {
            String key = entry.getKey();
            FileManage manage = entry.getValue();
            if ((type.equals(key) && location == null) || key.equals(SettingConstant.OSS_TYPE.getName(location))) {
                return manage;
            }
        }
        throw new HiverException("暂不支持该存储配置，请检查配置");
    }
}
