/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.rustfs.manage.impl;

import cn.hutool.core.date.DateTime;
import cn.hutool.core.date.DateUtil;
import cn.hutool.core.io.FileUtil;
import cn.hutool.core.util.StrUtil;
import com.google.gson.Gson;
import jakarta.annotation.Resource;
import jakarta.servlet.http.HttpServletResponse;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;
import org.springframework.web.multipart.MultipartFile;
import plus.hiver.common.constant.SettingConstant;
import plus.hiver.common.entity.Setting;
import plus.hiver.common.exception.HiverException;
import plus.hiver.common.service.SettingService;
import plus.hiver.module.system.vo.OssSetting;
import plus.hiver.rustfs.manage.FileManage;

import java.io.*;

/**
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
@Slf4j
@Component
public class LocalFileManage implements FileManage {
    @Resource
    private SettingService settingService;

    /**
     * 读取文件
     *
     * @param url 文件路径
     * @param response 响应
     */
    public static void view(String url, HttpServletResponse response) {
        File file = new File(url);
        if (!file.exists()) {
            throw new HiverException("文件不存在");
        }
        try (FileInputStream is = new FileInputStream(file);
             BufferedInputStream bis = new BufferedInputStream(is)) {
            OutputStream out = response.getOutputStream();
            byte[] buf = new byte[1024];
            int bytesRead;

            while ((bytesRead = bis.read(buf)) > 0) {
                out.write(buf, 0, bytesRead);
            }
        } catch (IOException e) {
            log.error(e.toString());
            throw new HiverException("读取文件出错");
        }
    }

    @Override
    public OssSetting getOssSetting() {
        Setting setting = settingService.findByKey(SettingConstant.OSS_TYPE.LOCAL_OSS.name());
        if (setting == null || StrUtil.isBlank(setting.getValue())) {
            throw new HiverException("您还未配置本地存储");
        }
        return new Gson().fromJson(setting.getValue(), OssSetting.class);
    }

    /**
     * 上传
     *
     * @param inputStream 输入流
     * @param key         文件名
     * @param file        文件
     * @return 结果
     */
    @Override
    public String inputStreamUpload(InputStream inputStream, String key, MultipartFile file) {
        OssSetting os = getOssSetting();
        DateTime date = DateUtil.date();
        String path = os.getFilePath() + "/" + date.year() + "/" + date.monthBaseOne() + "/" + date.dayOfMonth();
        File dir = new File(path);
        if (!dir.exists()) {
            dir.mkdirs();
        }
        File f = new File(path + "/" + key);
        if (f.exists()) {
            throw new HiverException("文件名已存在");
        }
        try {
            file.transferTo(f);
            return path + "/" + key;
        } catch (IOException e) {
            log.error(e.toString());
            throw new HiverException("上传文件出错");
        }
    }

    @Override
    public String inputStreamUpload(InputStream inputStream, String fileName) {
        OssSetting os = getOssSetting();
        DateTime date = DateUtil.date();
        String path = os.getFilePath() + "/" + date.year() + "/" + date.monthBaseOne() + "/" + date.dayOfMonth();
        File dir = new File(path);
        if (!dir.exists()) {
            dir.mkdirs();
        }
        File f = new File(path + "/" + fileName);
        if (f.exists()) {
            throw new HiverException("文件名已存在");
        }
        try {
            byte[] buffer = new byte[1024];
            OutputStream outputStream = new FileOutputStream(path + "/" + fileName);
            int bytesRead;
            while ((bytesRead = inputStream.read(buffer)) != -1) {
                outputStream.write(buffer, 0, bytesRead);
            }
            inputStream.close();
            outputStream.close();
            return path + "/" + fileName;
        } catch (IOException e) {
            log.error(e.toString());
            throw new HiverException("上传文件出错");
        }
    }

    /**
     * 注意此处需传入url
     *
     * @param url     原文件路径
     * @param toKey   新文件名
     * @return 新文件路径
     */
    @Override
    public String renameFile(String url, String toKey) {
        File old = new File(url);
        FileUtil.rename(old, toKey, false, true);
        return old.getParentFile() + "/" + toKey;
    }

    /**
     * 注意此处需传入url
     *
     * @param url    原文件路径
     * @param toKey  新文件名
     * @return 新文件路径
     */
    @Override
    public String copyFile(String url, String toKey) {
        File file = new File(url);
        String newUrl = file.getParentFile() + "/" + toKey;
        FileUtil.copy(file, new File(newUrl), true);
        return newUrl;
    }

    /**
     * 注意此处需传入url
     *
     * @param url 原文件路径
     */
    @Override
    public void deleteFile(String url) {
        FileUtil.del(new File(url));
    }
}
