/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.rustfs.manage.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.util.StrUtil;
import cn.hutool.json.JSONUtil;
import com.google.gson.Gson;
import jakarta.annotation.Resource;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;
import org.springframework.web.multipart.MultipartFile;
import plus.hiver.common.constant.SettingConstant;
import plus.hiver.common.entity.Setting;
import plus.hiver.common.exception.HiverException;
import plus.hiver.common.service.SettingService;
import plus.hiver.module.system.vo.OssSetting;
import plus.hiver.rustfs.dto.BucketPolicyConfigDto;
import plus.hiver.rustfs.dto.RustFSUploadResult;
import plus.hiver.rustfs.manage.FileManage;
import software.amazon.awssdk.core.sync.RequestBody;
import software.amazon.awssdk.services.s3.S3Client;
import software.amazon.awssdk.services.s3.model.*;

import java.io.*;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;

/**
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
@Slf4j
@Component
public class RustfsManage implements FileManage {
    @Resource
    private SettingService settingService;

    @Resource
    private S3Client s3Client;

    @Override
    public OssSetting getOssSetting() {
        Setting setting = settingService.findByKey(SettingConstant.OSS_TYPE.RUSTFS_OSS.name());
        if (setting == null || StrUtil.isBlank(setting.getValue())) {
            throw new HiverException("您还未配置阿里云OSS存储");
        }
        return new Gson().fromJson(setting.getValue(), OssSetting.class);
    }

    @Override
    public String inputStreamUpload(InputStream inputStream, String key, MultipartFile file) {
        OssSetting os = getOssSetting();
        // 判断Bucket是否存在
        if(!bucketExists(os.getBucket())){
            // 创建Bucket
            s3Client.createBucket(CreateBucketRequest.builder()
                    .bucket(os.getBucket())
                    .build());
            log.info("Bucket created: {}", os.getBucket());
            // 添加Bucket的访问策略
            String policy = JSONUtil.toJsonStr(createBucketPolicyConfigDto(os.getBucket()));
            log.info(policy);
            PutBucketPolicyRequest policyReq = PutBucketPolicyRequest.builder()
                    .bucket(os.getBucket())
                    .policy(policy)
                    .build();
            s3Client.putBucketPolicy(policyReq);
        }else{
            log.info("Bucket already exists.");
        }
        // 上传文件
        try {
            s3Client.putObject(PutObjectRequest.builder()
                    .bucket(os.getBucket())
                    .key(file.getOriginalFilename())
                    .contentType(file.getContentType())
                    .build(), RequestBody.fromInputStream(file.getInputStream(),file.getSize()));
            RustFSUploadResult uploadResult = new RustFSUploadResult();
            uploadResult.setName(file.getOriginalFilename());
            uploadResult.setUrl(os.getEndpoint() + "/" + key + "/" + file.getOriginalFilename());
            return uploadResult.getUrl();
        } catch (IOException e) {
            e.printStackTrace();
        }
        return null;
    }

    @Override
    public String inputStreamUpload(InputStream inputStream, String fileName) {
        OssSetting os = getOssSetting();
        // 判断Bucket是否存在
        if(!bucketExists(os.getBucket())){
            // 创建Bucket
            s3Client.createBucket(CreateBucketRequest.builder()
                    .bucket(os.getBucket())
                    .build());
            log.info("Bucket created: {}",os.getBucket());
            // 添加Bucket的访问策略
            String policy = JSONUtil.toJsonStr(createBucketPolicyConfigDto(os.getBucket()));
            log.info(policy);
            PutBucketPolicyRequest policyReq = PutBucketPolicyRequest.builder()
                    .bucket(os.getBucket())
                    .policy(policy)
                    .build();
            s3Client.putBucketPolicy(policyReq);
        }else{
            log.info("Bucket already exists.");
        }
        // 上传文件
        try {
            Path path = Paths.get(fileName);
            String contentType = Files.probeContentType(path);
            long contentLength = Files.size(path);
            String pureFileName = path.getFileName().toString();

            PutObjectRequest putObjectRequest = PutObjectRequest.builder()
                    .bucket(os.getBucket())
                    .key(fileName)
                    .contentType(contentType)
                    .contentLength(contentLength)
                    .build();
            s3Client.putObject(putObjectRequest, RequestBody.fromFile(path));
            RustFSUploadResult uploadResult = new RustFSUploadResult();
            uploadResult.setName(pureFileName);
            uploadResult.setUrl(os.getEndpoint() + "/" + os.getBucket() + "/" + pureFileName);
            return uploadResult.getUrl();
        } catch (IOException e) {
            e.printStackTrace();
        }
        return null;
    }

    @Override
    public String renameFile(String fromKey, String toKey) {
        OssSetting os = getOssSetting();
        // 检查源文件是否存在
        if (!objectExists(os.getBucket(), fromKey)) {
            throw new RuntimeException("源文件不存在: " + fromKey);
        }
        // 检查目标文件是否已存在
        if (objectExists(os.getBucket(), toKey)) {
            throw new RuntimeException("目标文件已存在: " + toKey);
        }
        // 执行复制+删除操作来模拟重命名
        copyFile(fromKey, toKey);
        deleteFile(fromKey);
        return toKey;
    }

    @Override
    public String copyFile(String fromKey, String toKey) {
        OssSetting os = getOssSetting();
        // 检查源文件是否存在
        if (!objectExists(os.getBucket(), fromKey)) {
            throw new RuntimeException("源文件不存在: " + fromKey);
        }
        CopyObjectRequest copyObjectRequest = CopyObjectRequest.builder()
                .sourceBucket(os.getBucket())
                .sourceKey(fromKey)
                .destinationBucket(os.getBucket())
                .destinationKey(toKey)
                .build();
        s3Client.copyObject(copyObjectRequest);
        return toKey;
    }

    @Override
    public void deleteFile(String key) {
        OssSetting os = getOssSetting();
        s3Client.deleteObject(DeleteObjectRequest.builder()
                .bucket(os.getBucket())
                .key(key)
                .build());
    }

    /**
     * 检查文件是否存在
     *
     * @param bucketName 存储桶名称
     * @param key 文件名
     * @return 是否存在
     */
    private boolean objectExists(String bucketName, String key) {
        try {
            s3Client.headObject(HeadObjectRequest.builder()
                    .bucket(bucketName)
                    .key(key)
                    .build());
            return true;
        } catch (S3Exception e) {
            if (e.statusCode() == 404) {
                return false;
            }
            throw e;
        }
    }

    /**
     * 判断Bucket是否存在
     */
    private boolean bucketExists(String bucketName) {
        try {
            s3Client.headBucket(request -> request.bucket(bucketName));
            return true;
        }
        catch (NoSuchBucketException exception) {
            return false;
        }
    }

    /**
     * 创建存储桶的访问策略，设置为只读权限
     */
    private BucketPolicyConfigDto createBucketPolicyConfigDto(String bucketName) {
        BucketPolicyConfigDto.Statement statement = BucketPolicyConfigDto.Statement.builder()
                .Effect("Allow")
                .Principal(BucketPolicyConfigDto.Principal.builder().AWS(new String[]{"*"}).build())
                .Action(new String[]{"s3:GetObject"})
                .Resource(new String[]{"arn:aws:s3:::"+bucketName+"/*"}).build();
        return BucketPolicyConfigDto.builder()
                .Version("2012-10-17")
                .Statement(CollUtil.toList(statement))
                .build();
    }
}
