/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.rustfs.serviceimpl;

import cn.hutool.core.util.StrUtil;
import jakarta.persistence.criteria.*;
import jakarta.transaction.Transactional;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.Sort;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.stereotype.Service;
import plus.hiver.common.base.HiverBaseServiceImpl;
import plus.hiver.common.constant.HiverConstant;
import plus.hiver.rustfs.dao.FileCategoryDao;
import plus.hiver.rustfs.entity.FileCategory;
import plus.hiver.rustfs.service.FileCategoryService;
import plus.hiver.rustfs.utils.FileUtil;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 文件分类接口实现
 *
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
@Slf4j
@Service
@Transactional
public class FileCategoryServiceImpl extends HiverBaseServiceImpl<FileCategory, Long, FileCategoryDao> implements FileCategoryService {
    public FileCategoryServiceImpl(FileCategoryDao dao) {
        super(dao);
    }

    @Override
    public List<FileCategory> findByCondition(FileCategory fileCategory, String sort, String order) {
        Sort.Direction d;
        if (StrUtil.isBlank(order)) {
            d = Sort.Direction.DESC;
        } else {
            d = Sort.Direction.valueOf(order.toUpperCase());
        }
        if (StrUtil.isBlank(sort)) {
            sort = "sortOrder";
        }
        List<FileCategory> list = dao.findAll(new Specification<FileCategory>() {
            @Override
            public Predicate toPredicate(Root<FileCategory> root, CriteriaQuery<?> cq, CriteriaBuilder cb) {
                Path<Long> parentIdField = root.get("parentId");
                Path<String> titleField = root.get("title");
                Path<Integer> delFlagField = root.get("deleted");
                Path<Boolean> isCollectField = root.get("isCollect");
                Path<String> createByField = root.get("createBy");

                List<Predicate> list = new ArrayList<>();

                if (fileCategory.getParentId() != null && fileCategory.getParentId() > 0) {
                    list.add(cb.equal(parentIdField, fileCategory.getParentId()));
                }

                if (StrUtil.isNotBlank(fileCategory.getTitle())) {
                    list.add(cb.like(titleField, "%" + fileCategory.getTitle() + "%"));
                }

                if (fileCategory.getDeleted() != null) {
                    list.add(cb.equal(delFlagField, fileCategory.getDeleted()));
                }
                if (fileCategory.getIsCollect() != null) {
                    list.add(cb.equal(isCollectField, fileCategory.getIsCollect()));
                }

                if (StrUtil.isNotBlank(fileCategory.getCreateBy())) {
                    list.add(cb.equal(createByField, fileCategory.getCreateBy()));
                }

                Predicate[] arr = new Predicate[list.size()];
                cq.where(list.toArray(arr));
                return null;
            }
        }, Sort.by(d, sort));

        if (HiverConstant.DEL_FLAG_TRASH.equals(fileCategory.getDeleted())) {
            list.forEach(e -> {
                if (FileUtil.isOvertime(e.getUpdateTime())) {
                    delete(e.getId());
                } else {
                    e.setRestDelTime(FileUtil.getRestTime(e.getUpdateTime()));
                }
            });
            list = list.stream().filter(fc -> !FileUtil.isOvertime(fc.getUpdateTime())).collect(Collectors.toList());
        }
        return list;
    }

    @Override
    public List<FileCategory> findByParentId(Long parentId, Integer deleted, String createBy) {
        FileCategory fileCategory = new FileCategory().setParentId(parentId);
        fileCategory.setDeleted(deleted);
        fileCategory.setCreateBy(createBy);
        return findByCondition(fileCategory, null, null);
    }

    @Override
    public List<FileCategory> findByTitleLikeAndCreateBy(String title, String createBy) {
        return dao.findByTitleLikeAndCreateByOrderBySortOrder(title, createBy);
    }
}
