/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.rustfs.service;

import org.springframework.data.domain.Page;
import plus.hiver.common.base.HiverBaseService;
import plus.hiver.common.vo.PageVo;
import plus.hiver.common.vo.SearchVo;
import plus.hiver.rustfs.entity.File;

/**
 * 文件管理接口
 *
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
public interface FileService extends HiverBaseService<File, Long> {
    /**
     * 多条件获取列表
     *
     * @param file        查询条件
     * @param searchVo    搜索条件
     * @param pageVo      分页条件
     * @param getCurrUser 是否获取当前用户
     * @return 结果
     */
    Page<File> getFileList(File file, SearchVo searchVo, PageVo pageVo, Boolean getCurrUser);

    /**
     * 回收站操作
     *
     * @param id          主键
     * @param getCurrUser 是否获取当前用户
     */
    void trash(Long id, Boolean getCurrUser);

    /**
     * 收藏
     *
     * @param id          主键
     * @param getCurrUser 是否获取当前用户
     */
    void collect(Long id, Boolean getCurrUser);

    /**
     * 重命名
     *
     * @param id          主键
     * @param newKey      新key
     * @param newTitle    新标题
     * @param getCurrUser 是否获取当前用户
     */
    void rename(Long id, String newKey, String newTitle, Boolean getCurrUser);

    /**
     * 创建副本
     *
     * @param id          主键
     * @param getCurrUser 是否获取当前用户
     */
    void copy(Long id, Boolean getCurrUser);

    /**
     * 删除
     *
     * @param ids         主键数组
     * @param getCurrUser 是否获取当前用户
     */
    void delete(Long[] ids, Boolean getCurrUser);

    /**
     * 清空回收站
     */
    void clearTrash();

    /**
     * 通过categoryId删除
     *
     * @param categoryId 分类id
     */
    void deleteByCategoryId(Long categoryId);
}
