/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.rustfs.serviceimpl;

import cn.hutool.core.date.DateUtil;
import cn.hutool.core.util.StrUtil;
import com.google.gson.Gson;
import jakarta.annotation.Nullable;
import jakarta.annotation.Resource;
import jakarta.persistence.EntityManager;
import jakarta.persistence.PersistenceContext;
import jakarta.persistence.criteria.*;
import jakarta.transaction.Transactional;
import lombok.extern.slf4j.Slf4j;
import org.springframework.cache.annotation.CacheConfig;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.stereotype.Service;
import plus.hiver.common.base.HiverBaseServiceImpl;
import plus.hiver.common.constant.HiverConstant;
import plus.hiver.common.constant.SettingConstant;
import plus.hiver.common.entity.User;
import plus.hiver.common.exception.HiverException;
import plus.hiver.common.redis.RedisTemplateHelper;
import plus.hiver.common.service.SettingService;
import plus.hiver.common.utils.CommonUtil;
import plus.hiver.common.utils.PageUtil;
import plus.hiver.common.utils.SecurityUtil;
import plus.hiver.common.vo.PageVo;
import plus.hiver.common.vo.SearchVo;
import plus.hiver.module.system.vo.OssSetting;
import plus.hiver.rustfs.dao.FileDao;
import plus.hiver.rustfs.entity.File;
import plus.hiver.rustfs.manage.FileManageFactory;
import plus.hiver.rustfs.service.FileService;
import plus.hiver.rustfs.utils.FileUtil;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * 文件管理接口实现
 *
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
@Slf4j
@Service
@Transactional
@CacheConfig(cacheNames = "file")
public class FileServiceImpl extends HiverBaseServiceImpl<File, Long, FileDao> implements FileService {
    @Resource
    private SecurityUtil securityUtil;

    @Resource
    private RedisTemplateHelper redisTemplateHelper;

    @Resource
    private SettingService settingService;

    @Resource
    private FileManageFactory fileManageFactory;

    @PersistenceContext
    private EntityManager entityManager;

    public FileServiceImpl(FileDao dao) {
        super(dao);
    }

    @Override
    @Cacheable(key = "#id")
    public File get(Long id) {
        // 避免缓存穿透
        String result = redisTemplateHelper.get("file::" + id);
        if ("null".equals(result)) {
            return null;
        }
        File file = dao.findById(id).orElse(null);
        if (file == null) {
            redisTemplateHelper.set("file::" + id, "null", 5L, TimeUnit.MINUTES);
        }
        return file;
    }

    @Override
    public Page<File> getFileList(File file, SearchVo searchVo, PageVo pageVo, Boolean getCurrUser) {
        if (getCurrUser) {
            file.setCreateBy(securityUtil.getCurrUserSimple().getUsername());
        }
        Page<File> page = dao.findAll(new Specification<File>() {
            @Nullable
            @Override
            public Predicate toPredicate(Root<File> root, CriteriaQuery<?> cq, CriteriaBuilder cb) {
                Path<String> titleField = root.get("title");
                Path<String> fKeyField = root.get("fKey");
                Path<String> typeField = root.get("type");
                Path<Long> categoryIdField = root.get("categoryId");
                Path<String> createByField = root.get("createBy");
                Path<Boolean> isCollectField = root.get("isCollect");
                Path<Integer> locationField = root.get("location");
                Path<Integer> delFlagField = root.get("deleted");
                Path<Date> createTimeField = root.get("createTime");

                List<Predicate> list = new ArrayList<>();

                // 模糊搜素
                if (StrUtil.isNotBlank(file.getTitle())) {
                    list.add(cb.like(titleField, '%' + file.getTitle() + '%'));
                }
                if (StrUtil.isNotBlank(file.getFKey())) {
                    list.add(cb.like(fKeyField, '%' + file.getFKey() + '%'));
                }
                if (StrUtil.isNotBlank(file.getType())) {
                    list.add(cb.like(typeField, '%' + file.getType() + '%'));
                }
                if (StrUtil.isNotBlank(file.getCreateBy())) {
                    list.add(cb.like(createByField, '%' + file.getCreateBy() + '%'));
                }

                if (file.getDeleted() != null) {
                    list.add(cb.equal(delFlagField, file.getDeleted()));
                }
                if (file.getLocation() != null) {
                    list.add(cb.equal(locationField, file.getLocation()));
                }
                if (file.getIsCollect() != null) {
                    list.add(cb.equal(isCollectField, file.getIsCollect()));
                }

                if (file.getCategoryId() != null && file.getCategoryId() > 0) {
                    list.add(cb.equal(categoryIdField, file.getCategoryId()));
                }

                // 创建时间
                if (StrUtil.isNotBlank(searchVo.getStartDate()) && StrUtil.isNotBlank(searchVo.getEndDate())) {
                    Date start = DateUtil.parse(searchVo.getStartDate());
                    Date end = DateUtil.parse(searchVo.getEndDate());
                    list.add(cb.between(createTimeField, start, DateUtil.endOfDay(end)));
                }

                Predicate[] arr = new Predicate[list.size()];
                cq.where(list.toArray(arr));
                return null;
            }
        }, PageUtil.initPage(pageVo));

        OssSetting os = new Gson().fromJson(settingService.findByKey(SettingConstant.OSS_TYPE.LOCAL_OSS.name()).getValue(), OssSetting.class);
        for (File e : page.getContent()) {
            if (e.getLocation() != null && e.getLocation().equals(SettingConstant.OSS_TYPE.LOCAL_OSS.ordinal())) {
                String url = os.getHttp() + os.getEndpoint() + "/";
                entityManager.detach(e);
                e.setUrl(url + e.getId());
            } else if(e.getLocation() != null && e.getLocation().equals(SettingConstant.OSS_TYPE.RUSTFS_OSS.ordinal())) {
                OssSetting rustfsOs = new Gson().fromJson(settingService.findByKey(SettingConstant.OSS_TYPE.RUSTFS_OSS.name()).getValue(), OssSetting.class);
                String url = rustfsOs.getHttp() + e.getUrl();
                entityManager.detach(e);
                e.setUrl(url);
            }
            if (HiverConstant.DEL_FLAG_TRASH.equals(file.getDeleted())) {
                if (FileUtil.isOvertime(e.getUpdateTime())) {
                    delete(new Long[]{e.getId()}, true);
                } else {
                    e.setRestDelTime(FileUtil.getRestTime(e.getUpdateTime()));
                }
            }
        }

        if (HiverConstant.DEL_FLAG_TRASH.equals(file.getDeleted())) {
            page = new PageImpl<>(page.getContent().stream().filter(e -> !FileUtil.isOvertime(e.getUpdateTime())).collect(Collectors.toList()),
                    page.getPageable(), page.getTotalPages());
        }
        return page;
    }

    @Override
    public void trash(Long id, Boolean getCurrUser) {
        File file = get(id);
        if (file.getLocation() == null) {
            throw new HiverException("存储位置未知");
        }

        if (getCurrUser) {
            User user = securityUtil.getCurrUserSimple();
            if (!user.getUsername().equals(file.getCreateBy())) {
                throw new HiverException("你无权操作此文件");
            }
        }
        file.setDeleted(HiverConstant.DEL_FLAG_TRASH.equals(file.getDeleted())
                ? HiverConstant.DEL_FLAG_FALSE : HiverConstant.DEL_FLAG_TRASH);
        dao.saveAndFlush(file);
    }

    @Override
    public void collect(Long id, Boolean getCurrUser) {
        File file = get(id);
        if (file.getLocation() == null) {
            throw new HiverException("存储位置未知");
        }

        if (getCurrUser) {
            User user = securityUtil.getCurrUserSimple();
            if (!user.getUsername().equals(file.getCreateBy())) {
                throw new HiverException("你无权操作此文件");
            }
        }
        file.setIsCollect(file.getIsCollect() == null || !file.getIsCollect());
        dao.saveAndFlush(file);
    }

    @Override
    @CacheEvict(key = "#id")
    public void rename(Long id, String newKey, String newTitle, Boolean getCurrUser) {
        File file = get(id);
        if (file.getLocation() == null) {
            throw new HiverException("存储位置未知");
        }

        if (getCurrUser) {
            User user = securityUtil.getCurrUserSimple();
            if (!user.getUsername().equals(file.getCreateBy())) {
                throw new HiverException("你无权操作此文件");
            }
        }

        String newUrl = "", oldKey = file.getFKey();
        if (StrUtil.isBlank(newKey)) {
            newKey = file.getFKey();
        }
        if (StrUtil.isBlank(newTitle)) {
            newTitle = file.getTitle();
        }

        if (!oldKey.equals(newKey)) {
            // 特殊处理本地服务器
            if (file.getLocation().equals(SettingConstant.OSS_TYPE.LOCAL_OSS.ordinal())) {
                oldKey = file.getUrl();
            }
            newUrl = fileManageFactory.getFileManage(file.getLocation()).renameFile(oldKey, newKey);
            file.setUrl(newUrl);
        }

        file.setTitle(newTitle).setFKey(newKey);
        update(file);
    }

    @Override
    public void copy(Long id, Boolean getCurrUser) {
        File file = get(id);
        if (file.getLocation() == null) {
            throw new HiverException("存储位置未知");
        }

        if (getCurrUser) {
            User user = securityUtil.getCurrUserSimple();
            if (!user.getUsername().equals(file.getCreateBy())) {
                throw new HiverException("你无权操作此文件");
            }
        }

        String key = file.getFKey();
        String toKey = CommonUtil.renamePic(key);
        // 特殊处理本地服务器
        if (file.getLocation().equals(SettingConstant.OSS_TYPE.LOCAL_OSS.ordinal())) {
            key = file.getUrl();
        }
        String newUrl = fileManageFactory.getFileManage(file.getLocation()).copyFile(key, toKey);
        File newFile = new File().setTitle("副本_" + file.getTitle()).setFKey(toKey).setSize(file.getSize()).setType(file.getType())
                .setLocation(file.getLocation()).setUrl(newUrl);
        if (getCurrUser) {
            newFile.setCategoryId(file.getCategoryId());
        }
        save(newFile);
    }

    @Override
    public void delete(Long[] ids, Boolean getCurrUser) {
        for (Long id : ids) {
            File file = get(id);
            if (file.getLocation() == null) {
                throw new HiverException("存储位置未知");
            }

            if (getCurrUser) {
                User user = securityUtil.getCurrUserSimple();
                if (!user.getUsername().equals(file.getCreateBy())) {
                    throw new HiverException("你无权操作此文件");
                }
            }

            // 特殊处理本地服务器
            String key = file.getFKey();
            if (file.getLocation().equals(SettingConstant.OSS_TYPE.LOCAL_OSS.ordinal())) {
                key = file.getUrl();
            }
            try {
                fileManageFactory.getFileManage(file.getLocation()).deleteFile(key);
            } catch (Exception e) {
                log.error("服务器删除文件失败，ID：" + file.getId() + " 存储位置：" + file.getLocation());
            }
            delete(id);
            redisTemplateHelper.delete("file::" + id);
        }
    }

    @Override
    public void clearTrash() {
        User user = securityUtil.getCurrUserSimple();
        dao.deleteByCreateByAndDelFlag(user.getUsername(), HiverConstant.DEL_FLAG_TRASH);
    }

    @Override
    public void deleteByCategoryId(Long categoryId) {
        dao.deleteByCategoryId(categoryId);
    }
}
