/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.rustfs.controller;

import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.annotation.Resource;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.bind.annotation.*;
import plus.hiver.common.annotation.PermissionTag;
import plus.hiver.common.api.Result;
import plus.hiver.common.constant.HiverConstant;
import plus.hiver.common.entity.User;
import plus.hiver.common.exception.HiverException;
import plus.hiver.common.redis.RedisTemplateHelper;
import plus.hiver.common.utils.CommonUtil;
import plus.hiver.common.utils.ResultUtil;
import plus.hiver.common.utils.SecurityUtil;
import plus.hiver.rustfs.entity.File;
import plus.hiver.rustfs.entity.FileCategory;
import plus.hiver.rustfs.service.FileCategoryService;
import plus.hiver.rustfs.service.FileService;

import java.util.List;

/**
 * 文件分类管理接口
 *
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
@Slf4j
@RestController
@Tag(name = "文件分类管理接口")
@PermissionTag(permission = "fileCategory:*" )
@AllArgsConstructor
@RequestMapping("/hiver/fileCategory")
@Transactional
public class FileCategoryController {
    @Resource
    private FileCategoryService fileCategoryService;

    @Resource
    private FileService fileService;

    @Resource
    private SecurityUtil securityUtil;

    @Resource
    private RedisTemplateHelper redisTemplateHelper;

    @GetMapping(value = "/getByCondition")
    @Operation(summary = "多条件获取")
    public Result<List<FileCategory>> getByCondition(@RequestParam(required = false) Long parentId,
                                                     @RequestParam(required = false) String title,
                                                     @RequestParam(required = false, defaultValue = "0") Integer delFlag,
                                                     @RequestParam(required = false) Boolean isCollect,
                                                     @RequestParam(required = false, defaultValue = "sortOrder") String sort,
                                                     @RequestParam(required = false, defaultValue = "asc") String order) {
        User user = securityUtil.getCurrUser();
        FileCategory fileCategory = new FileCategory().setParentId(parentId).setIsCollect(isCollect).setTitle(title);
        fileCategory.setCreateBy(user.getUsername());
        fileCategory.setDeleted(delFlag);
        List<FileCategory> list = fileCategoryService.findByCondition(fileCategory, sort, order);
        setInfo(list);
        return new ResultUtil<List<FileCategory>>().setData(list);
    }

    @PostMapping(value = "/trash")
    @Operation(summary = "用户文件夹回收站操作")
    public Result trash(@RequestParam Long id) {
        FileCategory fileCategory = fileCategoryService.get(id);
        User user = securityUtil.getCurrUserSimple();
        if (!user.getUsername().equals(fileCategory.getCreateBy())) {
            throw new HiverException("你无权操作此文件夹");
        }
        fileCategory.setDeleted(HiverConstant.DEL_FLAG_TRASH.equals(fileCategory.getDeleted())
                ? HiverConstant.DEL_FLAG_FALSE : HiverConstant.DEL_FLAG_TRASH);
        fileCategoryService.update(fileCategory);
        return ResultUtil.data(null);
    }

    @PostMapping(value = "/collect")
    @Operation(summary = "用户文件夹收藏")
    public Result collect(@RequestParam Long id) {
        FileCategory fileCategory = fileCategoryService.get(id);
        User user = securityUtil.getCurrUserSimple();
        if (!user.getUsername().equals(fileCategory.getCreateBy())) {
            throw new HiverException("你无权操作此文件夹");
        }
        fileCategory.setIsCollect(fileCategory.getIsCollect() == null ? true : !fileCategory.getIsCollect());
        return ResultUtil.data(null);
    }

    @PostMapping(value = "/rename")
    @Operation(summary = "用户文件夹重命名")
    public Result rename(@RequestParam Long id,
                         @RequestParam String newTitle) {
        FileCategory fileCategory = fileCategoryService.get(id);
        User user = securityUtil.getCurrUserSimple();
        if (!user.getUsername().equals(fileCategory.getCreateBy())) {
            throw new HiverException("你无权操作此文件夹");
        }
        fileCategory.setTitle(newTitle);
        return ResultUtil.data(null);
    }

    @PostMapping(value = "/add")
    @Operation(summary = "添加")
    public Result add(@RequestBody FileCategory fileCategory) {
        fileCategoryService.save(fileCategory);
        // 如果不是添加的一级 判断设置上级为父节点标识
        if (!HiverConstant.PARENT_ID.equals(fileCategory.getParentId())) {
            FileCategory parent = fileCategoryService.get(fileCategory.getParentId());
            if (parent.getIsParent() == null || !parent.getIsParent()) {
                parent.setIsParent(true);
                fileCategoryService.update(parent);
            }
        }
        return ResultUtil.success("添加成功");
    }

    @PostMapping(value = "/edit")
    @Operation(summary = "编辑")
    public Result edit(@RequestBody FileCategory fileCategory) {
        if (fileCategory.getId().equals(fileCategory.getParentId())) {
            return ResultUtil.error("上级节点不能为自己");
        }
        User user = securityUtil.getCurrUserSimple();
        FileCategory old = fileCategoryService.get(fileCategory.getId());
        if (!user.getUsername().equals(old.getCreateBy())) {
            return ResultUtil.error("你无权操作此文件夹");
        }
        Long oldParentId = old.getParentId();
        fileCategoryService.update(fileCategory);
        // 如果该节点不是一级节点 且修改了级别 判断上级还有无子节点
        if (!HiverConstant.PARENT_ID.equals(oldParentId) && !oldParentId.equals(fileCategory.getParentId())) {
            FileCategory parent = fileCategoryService.get(oldParentId);
            List<FileCategory> children = fileCategoryService
                    .findByParentId(parent.getId(), HiverConstant.DEL_FLAG_FALSE, user.getUsername());
            if (parent != null && (children == null || children.isEmpty())) {
                parent.setIsParent(false);
                fileCategoryService.update(parent);
            }
        }
        return ResultUtil.success("编辑成功");
    }

    @PostMapping(value = "/moveByIds")
    @Operation(summary = "批量修改分类（移动）")
    public Result moveByIds(@RequestParam Long[] ids,
                            @RequestParam Long categoryId) {
        User user = securityUtil.getCurrUser();
        if (!HiverConstant.PARENT_ID.equals(categoryId)) {
            FileCategory fileCategory = fileCategoryService.get(categoryId);
            if (!user.getUsername().equals(fileCategory.getCreateBy())) {
                return ResultUtil.error("你无权移动至该文件夹");
            }
        }
        for (Long id : ids) {
            File file = fileService.get(id);
            if (categoryId.equals(file.getCategoryId())) {
                // 分类没变化 无需移动
                continue;
            }
            if (!user.getUsername().equals(file.getCreateBy())) {
                return ResultUtil.error("你无权操作此文件");
            }
            file.setCategoryId(categoryId);
            fileService.save(file);
            redisTemplateHelper.delete("file::" + id);
        }
        return ResultUtil.success("批量移动文件成功");
    }

    @PostMapping(value = "/delByIds")
    @Operation(summary = "批量通过id删除")
    public Result delByIds(@RequestParam Long[] ids) {
        User user = securityUtil.getCurrUser();
        for (Long id : ids) {
            deleteRecursion(id, ids, user.getUsername());
        }
        return ResultUtil.success("批量通过id删除数据成功");
    }

    @GetMapping(value = "/clearTrash")
    @Operation(summary = "清空回收站")
    public Result clearTrash() {
        User user = securityUtil.getCurrUser();
        // 获取其回收站
        List<FileCategory> list = fileCategoryService
                .findByParentId(null, HiverConstant.DEL_FLAG_TRASH, user.getUsername());
        Long[] ids = list.stream().map(FileCategory::getId).toArray(Long[]::new);
        for (Long id : ids) {
            deleteRecursion(id, ids, user.getUsername());
        }
        return ResultUtil.success("清空文件夹回收站成功");
    }

    public void deleteRecursion(Long id, Long[] ids, String username) {
        // 获得其父节点
        FileCategory o = fileCategoryService.get(id);
        if (!username.equals(o.getCreateBy())) {
            throw new HiverException("你无权操作此文件夹");
        }
        FileCategory parent = null;
        if (o.getParentId() > 0) {
            parent = fileCategoryService.findById(o.getParentId());
        }
        fileCategoryService.delete(id);
        fileService.deleteByCategoryId(id);
        // 判断父节点是否还有子节点
        if (parent != null) {
            List<FileCategory> children = fileCategoryService.findByParentId(parent.getId(), null, username);
            if (children == null || children.isEmpty()) {
                parent.setIsParent(false);
                fileCategoryService.update(parent);
            }
        }
        // 递归删除
        List<FileCategory> objs = fileCategoryService.findByParentId(id, null, username);
        for (FileCategory obj : objs) {
            if (!CommonUtil.judgeIds(obj.getId(), ids)) {
                deleteRecursion(obj.getId(), ids, username);
            }
        }
    }

    @GetMapping(value = "/search")
    @Operation(summary = "名称模糊搜索")
    public Result<List<FileCategory>> searchByTitle(@RequestParam String title) {
        User user = securityUtil.getCurrUser();
        List<FileCategory> list = fileCategoryService.findByTitleLikeAndCreateBy("%" + title + "%", user.getUsername());
        setInfo(list);
        return new ResultUtil<List<FileCategory>>().setData(list);
    }

    public void setInfo(List<FileCategory> list) {
        // lambda表达式
        list.forEach(item -> {
            if (!HiverConstant.PARENT_ID.equals(item.getParentId())) {
                FileCategory parent = fileCategoryService.get(item.getParentId());
                item.setParentTitle(parent.getTitle());
            } else {
                item.setParentTitle("一级目录");
            }
        });
    }
}
