/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.rustfs.controller;

import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.annotation.Resource;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.Page;
import org.springframework.web.bind.annotation.*;
import plus.hiver.common.annotation.PermissionTag;
import plus.hiver.common.api.Result;
import plus.hiver.common.utils.ResultUtil;
import plus.hiver.common.vo.PageVo;
import plus.hiver.common.vo.SearchVo;
import plus.hiver.rustfs.entity.File;
import plus.hiver.rustfs.service.FileService;

/**
 * 文件管理管理接口
 *
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
@Slf4j
@RestController
@Tag(name = "文件管理管理接口")
@PermissionTag(permission = "file:*" )
@AllArgsConstructor
@RequestMapping("/hiver/file/user")
public class FileUserController {
    @Resource
    private FileService fileService;

    @GetMapping(value = "/getByCondition")
    @Operation(summary = "用户多条件分页获取")
    public Result<Page<File>> getFileList(File file, SearchVo searchVo, PageVo pageVo) {
        Page<File> page = fileService.getFileList(file, searchVo, pageVo, true);
        return new ResultUtil<Page<File>>().setData(page);
    }

    @PostMapping(value = "/trash")
    @Operation(summary = "用户文件回收站操作")
    public Result trash(@RequestParam Long[] ids) {
        for (Long id : ids) {
            fileService.trash(id, true);
        }
        return ResultUtil.data(null);
    }

    @PostMapping(value = "/collect")
    @Operation(summary = "用户文件收藏")
    public Result collect(@RequestParam Long id) {
        fileService.collect(id, true);
        return ResultUtil.data(null);
    }

    @PostMapping(value = "/rename")
    @Operation(summary = "用户文件重命名")
    public Result renameUserFile(@RequestParam Long id,
                                 @RequestParam String newTitle) {
        fileService.rename(id, null, newTitle, true);
        return ResultUtil.data(null);
    }

    @PostMapping(value = "/copy")
    @Operation(summary = "用户文件复制")
    public Result copy(@RequestParam Long id) {
        fileService.copy(id, true);
        return ResultUtil.data(null);
    }

    @PostMapping(value = "/delete")
    @Operation(summary = "用户文件删除")
    public Result delete(@RequestParam Long[] ids) {
        fileService.delete(ids, true);
        return ResultUtil.data(null);
    }

    @RequestMapping(value = "/clearTrash", method = RequestMethod.GET)
    @Operation(summary = "清空回收站")
    public Result clearTrash() {
        fileService.clearTrash();
        return ResultUtil.success("清空文件回收站成功");
    }
}
