/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.rustfs.controller;

import cn.hutool.core.util.StrUtil;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.annotation.Resource;
import jakarta.servlet.http.HttpServletResponse;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.web.bind.annotation.*;
import plus.hiver.common.annotation.PermissionTag;
import plus.hiver.common.exception.HiverException;
import plus.hiver.rustfs.entity.File;
import plus.hiver.rustfs.manage.impl.LocalFileManage;
import plus.hiver.rustfs.service.FileService;

import java.io.IOException;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;

/**
 * 文件管理管理接口
 *
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
@Slf4j
@RestController
@Tag(name = "文件管理管理接口")
@PermissionTag(permission = "file:*" )
@AllArgsConstructor
@RequestMapping("/hiver/file")
public class LocalFileViewController {
    @Resource
    private FileService fileService;

    @GetMapping(value = "/view/{id}")
    @Operation(summary = "本地存储预览文件")
    public void view(@PathVariable Long id,
                     @RequestParam(required = false) String filename,
                     @RequestParam(required = false, defaultValue = "false") Boolean preview,
                     @RequestParam(required = false, defaultValue = "UTF-8") String charset,
                     HttpServletResponse response) throws IOException {
        File file = fileService.get(id);
        if (file == null) {
            throw new HiverException("文件ID：" + id + "不存在");
        }
        if (StrUtil.isBlank(filename)) {
            filename = file.getFKey();
        }

        if (!preview) {
            response.addHeader("Content-Disposition", "attachment; filename=" + URLEncoder.encode(filename, StandardCharsets.UTF_8));
        }
        response.setContentLengthLong(file.getSize());
        response.setContentType(file.getType() + ";charset=" + charset);
        response.addHeader("Accept-Ranges", "bytes");
        if (file.getSize() != null && file.getSize() > 0) {
            response.addHeader("Content-Range", "bytes " + 0 + "-" + (file.getSize() - 1) + "/" + file.getSize());
        }
        LocalFileManage.view(file.getUrl(), response);
    }
}
