/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.rustfs.controller;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.json.JSONUtil;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;
import plus.hiver.common.annotation.PermissionTag;
import plus.hiver.common.api.Result;
import plus.hiver.common.utils.ResultUtil;
import plus.hiver.rustfs.dto.BucketPolicyConfigDto;
import plus.hiver.rustfs.dto.RustFSUploadResult;
import software.amazon.awssdk.core.sync.RequestBody;
import software.amazon.awssdk.services.s3.S3Client;
import software.amazon.awssdk.services.s3.model.*;

import java.io.IOException;

/**
 * RustFS对象存储管理Controller
 *
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
@Slf4j
@RestController
@Tag(name = "RustFS对象存储管理")
@PermissionTag(permission = "rustfs:*" )
@RequiredArgsConstructor
@RequestMapping("/hiver/rustfs")
public class RustFSController {
    @Autowired
    private S3Client s3Client;

    @Value("${rustfs.bucketName}")
    private String BUCKET_NAME;

    @Value("${rustfs.endpoint}")
    private String ENDPOINT;

    @Operation(summary = "文件上传")
    @RequestMapping(value = "/upload", method = RequestMethod.POST,consumes = MediaType.MULTIPART_FORM_DATA_VALUE)
    @ResponseBody
    public Result upload(@RequestPart("file") MultipartFile file) {
        // 判断Bucket是否存在
        if(!bucketExists(BUCKET_NAME)){
            // 创建Bucket
            s3Client.createBucket(CreateBucketRequest.builder()
                    .bucket(BUCKET_NAME)
                    .build());
            log.info("Bucket created: {}",BUCKET_NAME);
            // 添加Bucket的访问策略
            String policy = JSONUtil.toJsonStr(createBucketPolicyConfigDto(BUCKET_NAME));
            log.info(policy);
            PutBucketPolicyRequest policyReq = PutBucketPolicyRequest.builder()
                    .bucket(BUCKET_NAME)
                    .policy(policy)
                    .build();
            s3Client.putBucketPolicy(policyReq);
        }else{
            log.info("Bucket already exists.");
        }
        // 上传文件
        try {
            s3Client.putObject(PutObjectRequest.builder()
                    .bucket(BUCKET_NAME)
                    .key(file.getOriginalFilename())
                    .contentType(file.getContentType())
                    .build(), RequestBody.fromInputStream(file.getInputStream(),file.getSize()));
            RustFSUploadResult uploadResult = new RustFSUploadResult();
            uploadResult.setName(file.getOriginalFilename());
            uploadResult.setUrl(ENDPOINT + "/" + BUCKET_NAME + "/" + file.getOriginalFilename());
            return ResultUtil.data(uploadResult);
        } catch (IOException e) {
            e.printStackTrace();
        }
        return ResultUtil.error("文件上传失败");
    }

    @Operation(summary = "文件删除")
    @PostMapping(value = "/delete")
    @ResponseBody
    public Result delete(@RequestParam("objectName") String objectName) {
        // 删除对象
        s3Client.deleteObject(DeleteObjectRequest.builder()
                .bucket(BUCKET_NAME)
                .key(objectName)
                .build());
        return ResultUtil.success("删除成功");
    }

    /**
     * 判断Bucket是否存在
     */
    private boolean bucketExists(String bucketName) {
        try {
            s3Client.headBucket(request -> request.bucket(bucketName));
            return true;
        }
        catch (NoSuchBucketException exception) {
            return false;
        }
    }

    /**
     * 创建存储桶的访问策略，设置为只读权限
     */
    private BucketPolicyConfigDto createBucketPolicyConfigDto(String bucketName) {
        BucketPolicyConfigDto.Statement statement = BucketPolicyConfigDto.Statement.builder()
                .Effect("Allow")
                .Principal(BucketPolicyConfigDto.Principal.builder().AWS(new String[]{"*"}).build())
                .Action(new String[]{"s3:GetObject"})
                .Resource(new String[]{"arn:aws:s3:::"+bucketName+"/*"}).build();
        return BucketPolicyConfigDto.builder()
                .Version("2012-10-17")
                .Statement(CollUtil.toList(statement))
                .build();
    }
}
