/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.module.statistics.serviceimpl;

import cn.hutool.core.date.DateUtil;
import cn.hutool.core.util.StrUtil;
import jakarta.annotation.Nullable;
import jakarta.persistence.criteria.*;
import jakarta.transaction.Transactional;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.stereotype.Service;
import plus.hiver.common.base.HiverBaseServiceImpl;
import plus.hiver.common.vo.SearchVo;
import plus.hiver.module.statistics.dao.TrackEventDao;
import plus.hiver.module.statistics.entity.TrackEvent;
import plus.hiver.module.statistics.service.TrackEventService;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

/**
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
@Slf4j
@Service
@Transactional
public class TrackEventServiceImpl extends HiverBaseServiceImpl<TrackEvent, Long, TrackEventDao> implements TrackEventService {
    protected TrackEventServiceImpl(TrackEventDao dao) {
        super(dao);
    }

    @Override
    public Page<TrackEvent> findByCondition(TrackEvent trackEvent, SearchVo searchVo, Pageable pageable) {
        return dao.findAll(new Specification<TrackEvent>() {
            @Nullable
            @Override
            public Predicate toPredicate(Root<TrackEvent> root, CriteriaQuery<?> cq, CriteriaBuilder cb) {
                Path<String> ctField = root.get("ct");
                Path<String> klField = root.get("kl");
                Path<String> vlField = root.get("vl");
                Path<String> ttField = root.get("tt");
                Path<Date> createTimeField = root.get("createTime");
                List<Predicate> list = new ArrayList<>();
                // 模糊搜素
                if (StrUtil.isNotBlank(trackEvent.getCt())) {
                    list.add(cb.like(ctField, '%' + trackEvent.getCt() + '%'));
                }
                if (StrUtil.isNotBlank(trackEvent.getKl())) {
                    list.add(cb.like(klField, '%' + trackEvent.getKl() + '%'));
                }
                if (StrUtil.isNotBlank(trackEvent.getVl())) {
                    list.add(cb.like(vlField, '%' + trackEvent.getVl() + '%'));
                }
                if (StrUtil.isNotBlank(trackEvent.getTt())) {
                    list.add(cb.like(ttField, '%' + trackEvent.getTt() + '%'));
                }
                // 创建时间
                if (StrUtil.isNotBlank(searchVo.getStartDate()) && StrUtil.isNotBlank(searchVo.getEndDate())) {
                    Date start = DateUtil.parse(searchVo.getStartDate());
                    Date end = DateUtil.parse(searchVo.getEndDate());
                    list.add(cb.between(createTimeField, start, DateUtil.endOfDay(end)));
                }
                Predicate[] arr = new Predicate[list.size()];
                cq.where(list.toArray(arr));
                return null;
            }
        }, pageable);
    }
}
