/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.module.statistics.controller;

import cn.hutool.core.util.StrUtil;
import com.alibaba.fastjson.JSON;
import eu.bitwalker.useragentutils.Browser;
import eu.bitwalker.useragentutils.OperatingSystem;
import eu.bitwalker.useragentutils.UserAgent;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.annotation.Resource;
import jakarta.servlet.ServletOutputStream;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.codec.binary.Base64;
import org.springframework.data.domain.Page;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import plus.hiver.common.api.Result;
import plus.hiver.common.utils.DateUtil;
import plus.hiver.common.utils.IpInfoUtil;
import plus.hiver.common.utils.PageUtil;
import plus.hiver.common.utils.ResultUtil;
import plus.hiver.common.vo.PageVo;
import plus.hiver.common.vo.SearchVo;
import plus.hiver.module.statistics.entity.AppData;
import plus.hiver.module.statistics.entity.Statistics;
import plus.hiver.module.statistics.entity.TrackEvent;
import plus.hiver.module.statistics.httpmodel.StatisticsParms;
import plus.hiver.module.statistics.service.AppDataService;
import plus.hiver.module.statistics.service.StatisticsService;
import plus.hiver.module.statistics.service.TrackEventService;
import plus.hiver.module.statistics.utils.DecodeUtil;
import plus.hiver.module.statistics.utils.ImageUtil;

import javax.imageio.ImageIO;
import java.awt.image.BufferedImage;
import java.io.IOException;
import java.net.URLDecoder;
import java.nio.charset.StandardCharsets;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
@Slf4j
@RestController
@AllArgsConstructor
@Tag(name = "统计数据接口")
public class StatisticsController {
    @Resource
    private StatisticsService statisticsService;

    @Resource
    private TrackEventService trackEventService;

    @Resource
    private AppDataService appDataService;

    @PostMapping("/hiver/track/hm.gif")
    @Operation(summary = "应用事件统计")
    public void track(HttpServletRequest request, HttpServletResponse response, String data) {
        try {
            // 解决跨域
            request.setCharacterEncoding("UTF-8");
            response.setHeader("Access-Control-Allow-Origin", "*");
            response.setHeader("Access-Control-Allow-Methods", "POST, GET, OPTIONS, DELETE");
            response.setHeader("Access-Control-Max-Age", "3600");
            response.setHeader("Access-Control-Allow-Headers", "x-requested-with");
            response.setContentType("image/jpeg");

            if (StrUtil.isNotBlank(data)) {
                /**
                 *  对data=xxx，进行解码，先改处理%25的问题 URLDecoder.decode
                 */
                data = data.replaceAll("%(?![0-9a-fA-F]{2})", "%25");
                data = URLDecoder.decode(data, StandardCharsets.UTF_8);
                data = URLDecoder.decode(data, StandardCharsets.UTF_8);

                // base64解码
                byte[] encodeBase64 = Base64.decodeBase64(data.getBytes(StandardCharsets.UTF_8));
                String strdata = new String(encodeBase64, StandardCharsets.UTF_8);

                List<AppData> appDataList = JSON.parseArray(strdata, AppData.class);
                //保存数据后面修改，直接存库效率较低
                for (AppData appData : appDataList) {
                    appDataService.save(appData);
                }
            }

            //返回gif
            ServletOutputStream os = response.getOutputStream();
            BufferedImage im = ImageUtil.generateImageIo();
            ImageIO.write(im, "gif", os);
            os.close();
        } catch (IOException e) {
            log.error("统计数据异常", e);
        }
    }

    @GetMapping(value = {"/hiver/hm.gif"})
    @Operation(summary = "PV/UV数据统计")
    public void hm(HttpServletRequest request, HttpServletResponse response, StatisticsParms p, TrackEvent trackEvent) {
        try {
            // 解决跨域
            request.setCharacterEncoding("UTF-8");
            response.setHeader("Access-Control-Allow-Origin", "*");
            response.setHeader("Access-Control-Allow-Methods", "POST, GET, OPTIONS, DELETE");
            response.setHeader("Access-Control-Max-Age", "3600");
            response.setHeader("Access-Control-Allow-Headers", "x-requested-with");
            response.setContentType("image/jpeg");

            // 获取HTTP IP
            if (null != p || null != trackEvent) {
                String ipAddr = IpInfoUtil.getIpAddr(request);

                // 解码设置实体的值
                p.setTt(DecodeUtil.decode(p.getTt()));
                p.setSu(DecodeUtil.decode(DecodeUtil.decode(p.getSu())));
                p.setNu(DecodeUtil.decode(DecodeUtil.decode(p.getNu())));

                Statistics st = new Statistics();
                st.setAr(p.getAr());
                st.setAt(DateUtil.getDateMs()); // 时间
                st.setIp(ipAddr);
                st.setVi(p.getVi());
                st.setSu(p.getSu());
                st.setSi(p.getSi());
                st.setSb(p.getSb());
                st.setRnd(p.getRnd());
                st.setLn(p.getLn());
                st.setJa(p.getJa());
                st.setFl(p.getFl());
                st.setDs(p.getDs());
                st.setCl(p.getCl());
                st.setCk(p.getCk());
                st.setStag(p.getStag());
                st.setTt(p.getTt());
                st.setNu(p.getNu());
                st.setLt(p.getLt());
                st.setBid(p.getBid());
                st.setSid(p.getSid());
                st.setUa(p.getUa());

                /*
                  利用QQ IP纯正库解析IP 地址所属地区，
                 */
                if (StrUtil.isNotBlank(st.getIp())) {
                    String regex = "\\d{1,3}\\.\\d{1,3}\\.\\d{1,3}\\.\\d{1,3}";
                    Pattern pp = Pattern.compile(regex);
                    Matcher m = pp.matcher(st.getIp());
                    if (m.find()) {
                        String area = "";
                        String country = "";
                        try {
                            area = "";      //地区
                            country = "";   //国家
                            if (StrUtil.isNotBlank(area)) {
                                st.setAr(area);
                            }
                            if (StrUtil.isNotBlank(country)) {
                                st.setNetopt(country);
                            }
                        } catch (Exception e) {
                            log.error("统计数据异常", e);
                        }
                    }
                }

                // 浏览器识别，操作系统，设备类型
                if (!StrUtil.isBlank(st.getUa())) {
                    UserAgent userAgent = UserAgent.parseUserAgentString(st.getUa());
                    Browser browser = userAgent.getBrowser();
                    OperatingSystem os = userAgent.getOperatingSystem();
                    st.setBrowser(browser.getName());
                    st.setOs(os.getName());
                    st.setDevicetype(os.getDeviceType().getName());
                }

                /*
                  如果是事件
                 */
                if ((StrUtil.isNotBlank(trackEvent.getTag())) && (trackEvent.getTag().equals("_trackEvent"))) {
                    String ct = trackEvent.getCt();
                    ct = DecodeUtil.decode(ct);
                    String at = trackEvent.getAt();
                    at = DecodeUtil.decode(at);
                    String kl = trackEvent.getKl();
                    kl = DecodeUtil.decode(kl);
                    String vl = trackEvent.getVl();
                    vl = DecodeUtil.decode(vl);
                    trackEvent.setCt(ct);
                    trackEvent.setAt(at);
                    trackEvent.setKl(kl);
                    trackEvent.setVl(vl);
                    // 保存到 自定议事件表中
                    trackEventService.save(trackEvent);
                } else {
                    // 保存到自动流量中
                    statisticsService.save(st);
                }
            }
            //返回一张1px的gif图片  CODE=200
            ServletOutputStream os = response.getOutputStream();
            BufferedImage im = ImageUtil.generateImageIo();
            ImageIO.write(im, "gif", os);
            os.close();
        } catch (Exception e) {
            log.error("统计数据异常", e);
        }
    }

    @GetMapping(value = "/hiver/statistics/getByCondition")
    @Operation(summary = "获得PV/UV数据统计")
    public Result<Page<Statistics>> getStatisticsByCondition(Statistics statistics, SearchVo searchVo, PageVo pageVo) {
        Page<Statistics> page = statisticsService.findByCondition(statistics, searchVo, PageUtil.initPage(pageVo));
        return new ResultUtil<Page<Statistics>>().setData(page);
    }

    @PostMapping(value = "/hiver/statistics/delByIds")
    @Operation(summary = "通过id删除")
    public Result delAllStatisticsByIds(@RequestParam Long[] ids) {
        for (Long id : ids) {
            statisticsService.delete(id);
        }
        return ResultUtil.success("删除成功");
    }

    @GetMapping(value = "/hiver/track/getByCondition")
    @Operation(summary = "获得网页事件数据")
    public Result<Page<TrackEvent>> getByCondition(TrackEvent trackEvent, SearchVo searchVo, PageVo pageVo) {
        Page<TrackEvent> page = trackEventService.findByCondition(trackEvent, searchVo, PageUtil.initPage(pageVo));
        return new ResultUtil<Page<TrackEvent>>().setData(page);
    }

    @PostMapping(value = "/hiver/track/delByIds")
    @Operation(summary = "通过id删除")
    public Result delAllTrackByIds(@RequestParam Long[] ids) {
        for (Long id : ids) {
            trackEventService.delete(id);
        }
        return ResultUtil.success("删除成功");
    }

    @GetMapping(value = "/hiver/app/getByCondition")
    @Operation(summary = "获得应用事件数据")
    public Result<Page<AppData>> getByCondition(AppData appData, SearchVo searchVo, PageVo pageVo) {
        Page<AppData> page = appDataService.findByCondition(appData, searchVo, PageUtil.initPage(pageVo));
        return new ResultUtil<Page<AppData>>().setData(page);
    }

    @PostMapping(value = "/hiver/app/delByIds")
    @Operation(summary = "通过id删除")
    public Result delAllByIds(@RequestParam Long[] ids) {
        for (Long id : ids) {
            appDataService.delete(id);
        }
        return ResultUtil.success("删除成功");
    }
}
