/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.module.statistics.serviceimpl;

import cn.hutool.core.date.DateUtil;
import cn.hutool.core.util.StrUtil;
import jakarta.annotation.Nullable;
import jakarta.persistence.criteria.*;
import jakarta.transaction.Transactional;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.stereotype.Service;
import plus.hiver.common.base.HiverBaseServiceImpl;
import plus.hiver.common.vo.SearchVo;
import plus.hiver.module.statistics.dao.AppDataDao;
import plus.hiver.module.statistics.entity.AppData;
import plus.hiver.module.statistics.service.AppDataService;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

/**
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
@Slf4j
@Service
@Transactional
public class AppDataServiceImpl extends HiverBaseServiceImpl<AppData, Long, AppDataDao> implements AppDataService {
    protected AppDataServiceImpl(AppDataDao dao) {
        super(dao);
    }

    @Override
    public Page<AppData> findByCondition(AppData appData, SearchVo searchVo, Pageable pageable) {
        return dao.findAll(new Specification<AppData>() {
            @Nullable
            @Override
            public Predicate toPredicate(Root<AppData> root, CriteriaQuery<?> cq, CriteriaBuilder cb) {
                Path<String> appversionField = root.get("appversion");
                Path<String> imeiField = root.get("imei");
                Path<String> devicetokenField = root.get("devicetoken");
                Path<String> macaddrField = root.get("macaddr");
                Path<Date> createTimeField = root.get("createTime");
                List<Predicate> list = new ArrayList<>();
                // 模糊搜素
                if (StrUtil.isNotBlank(appData.getAppversion())) {
                    list.add(cb.like(appversionField, '%' + appData.getAppversion() + '%'));
                }
                if (StrUtil.isNotBlank(appData.getImei())) {
                    list.add(cb.like(imeiField, '%' + appData.getImei() + '%'));
                }
                if (StrUtil.isNotBlank(appData.getDevicetoken())) {
                    list.add(cb.like(devicetokenField, '%' + appData.getImei() + '%'));
                }
                if (StrUtil.isNotBlank(appData.getMacaddr())) {
                    list.add(cb.like(macaddrField, '%' + appData.getImei() + '%'));
                }
                // 创建时间
                if (StrUtil.isNotBlank(searchVo.getStartDate()) && StrUtil.isNotBlank(searchVo.getEndDate())) {
                    Date start = DateUtil.parse(searchVo.getStartDate());
                    Date end = DateUtil.parse(searchVo.getEndDate());
                    list.add(cb.between(createTimeField, start, DateUtil.endOfDay(end)));
                }
                Predicate[] arr = new Predicate[list.size()];
                cq.where(list.toArray(arr));
                return null;
            }
        }, pageable);
    }
}
