/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.module.statistics.serviceimpl;

import cn.hutool.core.date.DateUtil;
import cn.hutool.core.util.StrUtil;
import jakarta.annotation.Nullable;
import jakarta.persistence.criteria.*;
import jakarta.transaction.Transactional;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.stereotype.Service;
import plus.hiver.common.base.HiverBaseServiceImpl;
import plus.hiver.common.vo.SearchVo;
import plus.hiver.module.statistics.dao.StatisticsDao;
import plus.hiver.module.statistics.entity.Statistics;
import plus.hiver.module.statistics.service.StatisticsService;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

/**
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
@Slf4j
@Service
@Transactional
public class StatisticsServiceImpl extends HiverBaseServiceImpl<Statistics, Long, StatisticsDao> implements StatisticsService {
    protected StatisticsServiceImpl(StatisticsDao dao) {
        super(dao);
    }

    @Override
    public Page<Statistics> findByCondition(Statistics statistics, SearchVo searchVo, Pageable pageable) {
        return dao.findAll(new Specification<Statistics>() {
            @Nullable
            @Override
            public Predicate toPredicate(Root<Statistics> root, CriteriaQuery<?> cq, CriteriaBuilder cb) {
                Path<String> arField = root.get("ar");
                Path<String> ipField = root.get("ip");
                Path<String> ttField = root.get("tt");
                Path<String> nuField = root.get("nu");
                Path<Date> createTimeField = root.get("createTime");
                List<Predicate> list = new ArrayList<>();
                // 模糊搜素
                if (StrUtil.isNotBlank(statistics.getAr())) {
                    list.add(cb.like(arField, '%' + statistics.getAr() + '%'));
                }
                if (StrUtil.isNotBlank(statistics.getIp())) {
                    list.add(cb.like(ipField, '%' + statistics.getIp() + '%'));
                }
                if (StrUtil.isNotBlank(statistics.getTt())) {
                    list.add(cb.like(ttField, '%' + statistics.getTt() + '%'));
                }
                if (StrUtil.isNotBlank(statistics.getNu())) {
                    list.add(cb.like(nuField, '%' + statistics.getNu() + '%'));
                }
                // 创建时间
                if (StrUtil.isNotBlank(searchVo.getStartDate()) && StrUtil.isNotBlank(searchVo.getEndDate())) {
                    Date start = DateUtil.parse(searchVo.getStartDate());
                    Date end = DateUtil.parse(searchVo.getEndDate());
                    list.add(cb.between(createTimeField, start, DateUtil.endOfDay(end)));
                }
                Predicate[] arr = new Predicate[list.size()];
                cq.where(list.toArray(arr));
                return null;
            }
        }, pageable);
    }
}
