/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.module.system.controller;

import cn.hutool.core.util.StrUtil;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.annotation.Resource;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.cache.annotation.CacheConfig;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import plus.hiver.common.annotation.PermissionTag;
import plus.hiver.common.annotation.SystemLog;
import plus.hiver.common.api.Result;
import plus.hiver.common.config.properties.HiverTokenProperties;
import plus.hiver.common.constant.SecurityConstant;
import plus.hiver.common.entity.User;
import plus.hiver.common.enums.LogType;
import plus.hiver.common.redis.RedisTemplateHelper;
import plus.hiver.common.utils.ResultUtil;
import plus.hiver.common.utils.SecurityUtil;

import java.util.concurrent.TimeUnit;

/**
 * 用户认证相关接口
 *
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
@Slf4j
@RestController
@AllArgsConstructor
@Tag(name = "用户认证相关接口")
@PermissionTag(permission = "auth:*" )
@RequestMapping("/hiver/auth")
@CacheConfig(cacheNames = "user")
@Transactional
public class AuthController {
    public static final String USER = "username::";

    public static final String LOGIN_FAIL_FLAG = "LOGIN_FAIL_FLAG:";

    public static final String LOGIN_TIME_LIMIT = "LOGIN_TIME_LIMIT:";

    public static final Integer LOGIN_FAIL_TIP_TIME = 3;

    @Resource
    private HiverTokenProperties tokenProperties;

    @Resource
    private SecurityUtil securityUtil;

    @Resource
    private RedisTemplateHelper redisTemplateHelper;

    @PostMapping(value = "/login")
    @SystemLog(description = "账号登录", type = LogType.LOGIN)
    @Operation(summary = "账号/手机/邮箱登录")
    public Result login(@RequestParam String username,
                        @RequestParam String password,
                        @RequestParam(required = false) Boolean saveLogin) {
        String loginFailKey = LOGIN_FAIL_FLAG + username;
        String loginTimeKey = LOGIN_TIME_LIMIT + username;

        String valueFailFlag = redisTemplateHelper.get(loginFailKey);
        Long timeRest = redisTemplateHelper.getExpire(loginFailKey, TimeUnit.MINUTES);
        if (StrUtil.isNotBlank(valueFailFlag)) {
            // 超过限制次数
            return ResultUtil.error("登录错误次数超过限制，请" + timeRest + "分钟后再试");
        }
        User user = securityUtil.checkUserPassword(username, password);
        if (user == null) {
            // 记录密码错误次数
            String valueTime = redisTemplateHelper.get(loginTimeKey);
            if (StrUtil.isBlank(valueTime)) {
                valueTime = "0";
            }
            // 获取已登录错误次数
            Integer loginFailTime = Integer.parseInt(valueTime) + 1;
            redisTemplateHelper.set(loginTimeKey, loginFailTime.toString(), tokenProperties.getLoginAfterTime(), TimeUnit.MINUTES);
            if (loginFailTime >= tokenProperties.getLoginTimeLimit()) {
                redisTemplateHelper.set(loginFailKey, "FAIL", tokenProperties.getLoginAfterTime(), TimeUnit.MINUTES);
            }
            int restLoginTime = tokenProperties.getLoginTimeLimit() - loginFailTime;
            if (restLoginTime > 0 && restLoginTime <= LOGIN_FAIL_TIP_TIME) {
                return ResultUtil.error("账号或密码错误，还有" + restLoginTime + "次尝试机会");
            } else if (restLoginTime <= 0) {
                return ResultUtil.error("登录错误次数超过限制，请" + tokenProperties.getLoginAfterTime() + "分钟后再试");
            }
            return ResultUtil.error("账号或密码错误");
        }
        String accessToken = securityUtil.getToken(user, saveLogin);
        return ResultUtil.data(accessToken);
    }

    @PostMapping(value = "/logout")
    @Operation(summary = "用户注销")
    public Result logout() {
        User user = securityUtil.getCurrUser();
        // 删除个人信息
        redisTemplateHelper.delete(USER + user.getUsername());
        // 获得令牌
        String token = redisTemplateHelper.get(SecurityConstant.USER_TOKEN + user.getUsername());
        // 删除在线状态
        redisTemplateHelper.delete(SecurityConstant.ONLINE_USER_PRE + user.getUsername() + ":" + token);
        // 交互token前缀key
        redisTemplateHelper.delete(SecurityConstant.TOKEN_PRE + token);
        // 删除令牌
        redisTemplateHelper.delete(SecurityConstant.USER_TOKEN + user.getUsername());
        return ResultUtil.success("注销成功");
    }
}
