/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.module.system.controller;

import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.annotation.Resource;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.bind.annotation.*;
import plus.hiver.common.annotation.PermissionTag;
import plus.hiver.common.api.Result;
import plus.hiver.common.redis.RedisTemplateHelper;
import plus.hiver.common.utils.ResultUtil;
import plus.hiver.module.system.entity.Dict;
import plus.hiver.module.system.service.DictDataService;
import plus.hiver.module.system.service.DictService;

import java.util.List;

/**
 * 字典管理接口
 *
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
@Slf4j
@RestController
@AllArgsConstructor
@Tag(name = "字典管理接口")
@PermissionTag(permission = "dict:*" )
@RequestMapping("/hiver/dict")
@Transactional
public class DictController {
    @Resource
    private DictService dictService;

    @Resource
    private DictDataService dictDataService;

    @Resource
    private RedisTemplateHelper redisTemplateHelper;

    @GetMapping(value = "/getAll")
    @Operation(summary = "获取全部数据")
    public Result<List<Dict>> getAll() {
        List<Dict> list = dictService.findAllOrderBySortOrder();
        return new ResultUtil<List<Dict>>().setData(list);
    }

    @PostMapping(value = "/add")
    @Operation(summary = "添加")
    public Result add(Dict dict) {
        if (dictService.findByType(dict.getType()) != null) {
            return ResultUtil.error("字典类型Type已存在");
        }
        dictService.save(dict);
        return ResultUtil.success("添加成功");
    }

    @PostMapping(value = "/edit")
    @Operation(summary = "编辑")
    public Result edit(Dict dict) {
        Dict old = dictService.get(dict.getId());
        // 若type修改判断唯一
        if (!old.getType().equals(dict.getType()) && dictService.findByType(dict.getType()) != null) {
            return ResultUtil.error("字典类型Type已存在");
        }
        dictService.update(dict);
        // 删除缓存
        redisTemplateHelper.delete("dictData::" + dict.getType());
        return ResultUtil.success("编辑成功");
    }

    @PostMapping(value = "/delByIds")
    @Operation(summary = "通过id删除")
    public Result delAllByIds(@RequestParam Long[] ids) {
        for (Long id : ids) {
            Dict dict = dictService.get(id);
            dictService.delete(id);
            dictDataService.deleteByDictId(id);
            // 删除缓存
            redisTemplateHelper.delete("dictData::" + dict.getType());
        }
        return ResultUtil.success("删除成功");
    }

    @RequestMapping(value = "/search", method = RequestMethod.GET)
    @Operation(summary = "搜索字典")
    public Result<List<Dict>> searchPermissionList(@RequestParam String key) {
        List<Dict> list = dictService.findByTitleOrTypeLike(key);
        return new ResultUtil<List<Dict>>().setData(list);
    }
}
