/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.module.system.controller;

import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.annotation.Resource;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.cache.annotation.CacheConfig;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.data.domain.Page;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.bind.annotation.*;
import plus.hiver.common.annotation.PermissionTag;
import plus.hiver.common.api.Result;
import plus.hiver.common.redis.RedisTemplateHelper;
import plus.hiver.common.utils.PageUtil;
import plus.hiver.common.utils.ResultUtil;
import plus.hiver.common.vo.PageVo;
import plus.hiver.module.system.entity.Dict;
import plus.hiver.module.system.entity.DictData;
import plus.hiver.module.system.service.DictDataService;
import plus.hiver.module.system.service.DictService;

import java.util.List;

/**
 * 字典数据接口
 *
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
@Slf4j
@RestController
@AllArgsConstructor
@Tag(name = "字典数据接口")
@PermissionTag(permission = "dictData:*" )
@RequestMapping("/hiver/dictData")
@CacheConfig(cacheNames = "dictData")
@Transactional
public class DictDataController {
    @Resource
    private DictService dictService;

    @Resource
    private DictDataService dictDataService;

    @Resource
    private RedisTemplateHelper redisTemplateHelper;

    @GetMapping(value = "/getByCondition")
    @Operation(summary = "多条件分页获取用户列表")
    public Result<Page<DictData>> getByCondition(DictData dictData,
                                                 PageVo pageVo) {
        Page<DictData> page = dictDataService.findByCondition(dictData, PageUtil.initPage(pageVo));
        return new ResultUtil<Page<DictData>>().setData(page);
    }

    @GetMapping(value = "/getByType/{type}")
    @Operation(summary = "通过类型获取")
    @Cacheable(key = "#type")
    public Result getByType(@PathVariable String type) {
        Dict dict = dictService.findByType(type);
        if (dict == null) {
            return ResultUtil.error("字典类型 " + type + " 不存在");
        }
        List<DictData> list = dictDataService.findByDictId(dict.getId());
        return ResultUtil.data(list);
    }

    @PostMapping(value = "/add")
    @Operation(summary = "添加")
    public Result add(DictData dictData) {
        Dict dict = dictService.get(dictData.getDictId());
        dictDataService.save(dictData);
        // 删除缓存
        redisTemplateHelper.delete("dictData::" + dict.getType());
        return ResultUtil.success("添加成功");
    }

    @PostMapping(value = "/edit")
    @Operation(summary = "编辑")
    public Result edit(DictData dictData) {
        dictDataService.update(dictData);
        // 删除缓存
        Dict dict = dictService.get(dictData.getDictId());
        redisTemplateHelper.delete("dictData::" + dict.getType());
        return ResultUtil.success("编辑成功");
    }

    @PostMapping(value = "/delByIds")
    @Operation(summary = "批量通过id删除")
    public Result delByIds(@RequestParam Long[] ids) {
        for (Long id : ids) {
            DictData dictData = dictDataService.get(id);
            Dict dict = dictService.get(dictData.getDictId());
            dictDataService.delete(id);
            // 删除缓存
            redisTemplateHelper.delete("dictData::" + dict.getType());
        }
        return ResultUtil.success("批量通过id删除数据成功");
    }
}
