/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.module.system.controller;

import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.annotation.Resource;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.Page;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.bind.annotation.*;
import plus.hiver.common.annotation.PermissionTag;
import plus.hiver.common.api.Result;
import plus.hiver.common.entity.Log;
import plus.hiver.common.service.LogService;
import plus.hiver.common.utils.PageUtil;
import plus.hiver.common.utils.ResultUtil;
import plus.hiver.common.vo.PageVo;
import plus.hiver.common.vo.SearchVo;

import java.io.UnsupportedEncodingException;

/**
 * 日志管理接口
 *
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
@Slf4j
@RestController
@AllArgsConstructor
@Tag(name = "日志管理接口")
@PermissionTag(permission = "log:*" )
@RequestMapping("/hiver/log")
@Transactional
public class LogController {
    @Resource
    private LogService logService;

    @GetMapping(value = "/getAllByPage")
    @Operation(summary = "分页获取全部")
    public Result getAllByPage(@RequestParam(required = false) Integer type,
                               @RequestParam String key,
                               SearchVo searchVo,
                               PageVo pageVo) throws UnsupportedEncodingException {
        Page<Log> log = logService.findByConfition(type, key, searchVo, PageUtil.initPage(pageVo));
        return ResultUtil.data(log);
    }

    @PostMapping(value = "/delByIds")
    @Operation(summary = "批量删除")
    public Result delByIds(@RequestParam Long[] ids) {
        for (Long id : ids) {
            logService.delete(id);
        }
        return ResultUtil.success("删除成功");
    }

    @PostMapping(value = "/delAll")
    @Operation(summary = "全部删除")
    public Result delAll() {
        logService.deleteAll();
        return ResultUtil.success("删除成功");
    }
}
