/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.module.system.controller;

import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.annotation.Resource;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.Page;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.bind.annotation.*;
import plus.hiver.common.annotation.PermissionTag;
import plus.hiver.common.api.Result;
import plus.hiver.common.constant.HiverConstant;
import plus.hiver.common.entity.Role;
import plus.hiver.common.entity.RoleDepartment;
import plus.hiver.common.entity.RolePermission;
import plus.hiver.common.entity.UserRole;
import plus.hiver.common.redis.RedisTemplateHelper;
import plus.hiver.common.service.RoleDepartmentService;
import plus.hiver.common.service.RolePermissionService;
import plus.hiver.common.service.RoleService;
import plus.hiver.common.service.UserRoleService;
import plus.hiver.common.utils.PageUtil;
import plus.hiver.common.utils.ResultUtil;
import plus.hiver.common.vo.PageVo;

import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 角色管理接口
 *
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
@Slf4j
@RestController
@AllArgsConstructor
@Tag(name = "角色管理接口")
@PermissionTag(permission = "role:*" )
@RequestMapping("/hiver/role")
@Transactional
public class RoleController {
    @Resource
    private RoleService roleService;

    @Resource
    private UserRoleService userRoleService;

    @Resource
    private RolePermissionService rolePermissionService;

    @Resource
    private RoleDepartmentService roleDepartmentService;

    @Resource
    private RedisTemplateHelper redisTemplateHelper;

    @GetMapping(value = "/getAllList")
    @Operation(summary = "获取全部角色")
    public Result roleGetAll() {
        List<Role> list = roleService.getAll();
        return ResultUtil.data(list);
    }

    @GetMapping(value = "/getAllByPage")
    @Operation(summary = "分页获取角色")
    public Result<Page<Role>> getRoleByPage(String key, PageVo page) {
        Page<Role> list = roleService.findByCondition(key, PageUtil.initPage(page));
        for (Role role : list.getContent()) {
            // 角色拥有权限
            List<RolePermission> permissions = rolePermissionService.findByRoleId(role.getId());
            role.setPermissions(permissions);
            // 角色拥有数据权限
            List<RoleDepartment> departments = roleDepartmentService.findByRoleId(role.getId());
            role.setDepartments(departments);
        }
        return new ResultUtil<Page<Role>>().setData(list);
    }

    @PostMapping(value = "/setDefault")
    @Operation(summary = "设置或取消默认角色")
    public Result setDefault(@RequestParam Long id,
                             @RequestParam Boolean isDefault) {
        Role role = roleService.get(id);
        if (role == null) {
            return ResultUtil.error("角色不存在");
        }
        role.setDefaultRole(isDefault);
        roleService.update(role);
        return ResultUtil.success("设置成功");
    }

    @PostMapping(value = "/editRolePerm")
    @Operation(summary = "编辑角色分配菜单权限")
    public Result editRolePerm(@RequestParam Long roleId,
                               @RequestParam(required = false) Long[] permIds) {
        // 删除其关联权限
        rolePermissionService.deleteByRoleId(roleId);
        // 批量分配新权限
        if (permIds != null) {
            List<RolePermission> list = Arrays.asList(permIds).stream().map(e -> {
                return new RolePermission().setRoleId(roleId).setPermissionId(e);
            }).collect(Collectors.toList());
            rolePermissionService.saveOrUpdateAll(list);
        }
        // 手动批量删除缓存
        redisTemplateHelper.deleteByPattern("user:*");
        redisTemplateHelper.deleteByPattern("userRole:*");
        redisTemplateHelper.deleteByPattern("permission::userMenuList:*");
        return ResultUtil.data(null);
    }

    @PostMapping(value = "/editRoleDep")
    @Operation(summary = "编辑角色分配数据权限")
    public Result editRoleDep(@RequestParam Long roleId,
                              @RequestParam Integer dataType,
                              @RequestParam(required = false) Long[] depIds) {
        Role r = roleService.get(roleId);
        r.setDataType(dataType);
        roleService.update(r);
        if (HiverConstant.DATA_TYPE_CUSTOM.equals(dataType)) {
            // 删除其关联数据权限
            roleDepartmentService.deleteByRoleId(roleId);
            // 批量分配新数据权限
            if (depIds != null) {
                List<RoleDepartment> list = Arrays.asList(depIds).stream().map(e -> {
                    return new RoleDepartment().setRoleId(roleId).setDepartmentId(e);
                }).collect(Collectors.toList());
                roleDepartmentService.saveOrUpdateAll(list);
            }
        }
        // 手动删除相关缓存
        redisTemplateHelper.deleteByPattern("department:*");
        redisTemplateHelper.deleteByPattern("userRole:*");
        return ResultUtil.data(null);
    }

    @PostMapping(value = "/save")
    @Operation(summary = "保存数据")
    public Result<Role> save(Role role) {
        Role r = roleService.save(role);
        return new ResultUtil<Role>().setData(r);
    }

    @PostMapping(value = "/edit")
    @Operation(summary = "更新数据")
    public Result<Role> edit(Role entity) {
        Role r = roleService.update(entity);
        // 手动批量删除缓存
        redisTemplateHelper.deleteByPattern("user:*");
        redisTemplateHelper.deleteByPattern("userRole:*");
        return new ResultUtil<Role>().setData(r);
    }

    @PostMapping(value = "/delByIds")
    @Operation(summary = "批量通过ids删除")
    public Result delByIds(@RequestParam Long[] ids) {
        for (Long id : ids) {
            List<UserRole> list = userRoleService.findByRoleId(id);
            if (list != null && list.size() > 0) {
                return ResultUtil.error("删除失败，包含正被用户使用关联的角色");
            }
        }
        for (Long id : ids) {
            roleService.delete(id);
            // 删除关联菜单权限
            rolePermissionService.deleteByRoleId(id);
            // 删除关联数据权限
            roleDepartmentService.deleteByRoleId(id);
        }
        return ResultUtil.success("批量通过id删除数据成功");
    }
}
