/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.module.system.serviceimpl;

import cn.hutool.core.util.StrUtil;
import jakarta.annotation.Nullable;
import jakarta.persistence.criteria.*;
import jakarta.transaction.Transactional;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.stereotype.Service;
import plus.hiver.common.base.HiverBaseServiceImpl;
import plus.hiver.common.constant.HiverConstant;
import plus.hiver.module.system.dao.DictDataDao;
import plus.hiver.module.system.entity.DictData;
import plus.hiver.module.system.service.DictDataService;

import java.util.ArrayList;
import java.util.List;

/**
 * 字典数据接口实现
 *
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
@Slf4j
@Service
@Transactional
public class DictDataServiceImpl extends HiverBaseServiceImpl<DictData, Long, DictDataDao> implements DictDataService {
    public DictDataServiceImpl(DictDataDao dao) {
        super(dao);
    }

    @Override
    public Page<DictData> findByCondition(DictData dictData, Pageable pageable) {
        return dao.findAll(new Specification<DictData>() {
            @Nullable
            @Override
            public Predicate toPredicate(Root<DictData> root, CriteriaQuery<?> cq, CriteriaBuilder cb) {
                Path<String> titleField = root.get("title");
                Path<Integer> statusField = root.get("status");
                Path<Long> dictIdField = root.get("dictId");

                List<Predicate> list = new ArrayList<>();

                // 模糊搜素
                if (StrUtil.isNotBlank(dictData.getTitle())) {
                    list.add(cb.like(titleField, '%' + dictData.getTitle() + '%'));
                }

                // 状态
                if (dictData.getStatus() != null) {
                    list.add(cb.equal(statusField, dictData.getStatus()));
                }

                // 所属字典
                if (dictData.getDictId() > 0) {
                    list.add(cb.equal(dictIdField, dictData.getDictId()));
                }

                Predicate[] arr = new Predicate[list.size()];
                cq.where(list.toArray(arr));
                return null;
            }
        }, pageable);
    }

    @Override
    public List<DictData> findByDictId(Long dictId) {
        return dao.findByDictIdAndStatusOrderBySortOrder(dictId, HiverConstant.STATUS_NORMAL);
    }

    @Override
    public void deleteByDictId(Long dictId) {
        dao.deleteByDictId(dictId);
    }

    @Override
    public DictData findByDictIdAndValue(Long dictId, String value) {
        return dao.findByDictIdAndValue(dictId, value);
    }
}
