/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.module.system.serviceimpl;

import com.google.gson.Gson;
import jakarta.annotation.Nullable;
import jakarta.persistence.criteria.*;
import jakarta.transaction.Transactional;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.messaging.simp.SimpMessagingTemplate;
import org.springframework.stereotype.Service;
import plus.hiver.common.base.HiverBaseServiceImpl;
import plus.hiver.common.constant.MessageConstant;
import plus.hiver.common.exception.HiverException;
import plus.hiver.module.system.dao.MessageDao;
import plus.hiver.module.system.dao.MessageSendDao;
import plus.hiver.module.system.entity.Message;
import plus.hiver.module.system.entity.MessageSend;
import plus.hiver.module.system.service.MessageSendService;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * 消息发送接口实现
 *
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
@Slf4j
@Service
@Transactional
public class MessageSendServiceImpl extends HiverBaseServiceImpl<MessageSend, Long, MessageSendDao> implements MessageSendService {
    @Autowired
    private MessageDao messageDao;

    @Autowired
    private SimpMessagingTemplate messagingTemplate;

    public MessageSendServiceImpl(MessageSendDao dao) {
        super(dao);
    }

    @Override
    public MessageSend send(MessageSend messageSend) {
        MessageSend ms = dao.save(messageSend);
        messagingTemplate.convertAndSendToUser(messageSend.getUserId().toString(), "/queue/subscribe", "您收到了新的消息");
        return ms;
    }

    @Override
    public void deleteByMessageId(Long messageId) {
        dao.deleteByMessageId(messageId);
    }

    @Override
    public Page<MessageSend> findByCondition(MessageSend messageSend, Pageable pageable) {
        return dao.findAll(new Specification<MessageSend>() {
            @Nullable
            @Override
            public Predicate toPredicate(Root<MessageSend> root, CriteriaQuery<?> cq, CriteriaBuilder cb) {
                Path<Long> messageIdField = root.get("messageId");
                Path<Long> userIdField = root.get("userId");
                Path<Integer> statusField = root.get("status");

                List<Predicate> list = new ArrayList<>();

                if (messageSend.getMessageId() > 0) {
                    list.add(cb.equal(messageIdField, messageSend.getMessageId()));
                }
                if (messageSend.getUserId() > 0) {
                    list.add(cb.equal(userIdField, messageSend.getUserId()));
                }
                if (messageSend.getStatus() != null) {
                    list.add(cb.equal(statusField, messageSend.getStatus()));
                }

                Predicate[] arr = new Predicate[list.size()];
                cq.where(list.toArray(arr));
                return null;
            }
        }, pageable);
    }

    @Override
    public void updateStatusByUserId(Long userId, Integer status) {
        dao.updateStatusByUserId(userId, status);
    }

    @Override
    public void deleteByUserId(Long userId) {
        dao.deleteByUserId(userId, MessageConstant.MESSAGE_STATUS_READ);
    }

    @Override
    public Message getTemplateMessage(Long messageId, Map<String, String> params) {
        Message message = messageDao.findById(messageId).orElse(null);
        if (message == null) {
            throw new HiverException("消息ID：" + messageId + "不存在");
        }
        // 放入变量
        Message newMessage = new Message();
        newMessage.setTitle(changeParams(message.getTitle(), params));
        newMessage.setContent(changeParams(message.getContent(), params));
        return newMessage;
    }

    @Override
    public void sendTemplateMessage(List<Long> userIds, Long messageId, Map<String, String> params) {
        Message message = messageDao.findById(messageId).orElse(null);
        if (message == null) {
            throw new HiverException("消息ID：" + messageId + "不存在");
        }
        List<MessageSend> messageSends = new ArrayList<>();
        for (Long userId : userIds) {
            MessageSend ms = new MessageSend();
            ms.setUserId(userId).setMessageId(messageId);
            // 放入变量
            ms.setTitle(changeParams(message.getTitle(), params));
            ms.setContent(changeParams(message.getContent(), params));
            ms.setParams(new Gson().toJson(params));
            messageSends.add(ms);
            messagingTemplate.convertAndSendToUser(userId.toString(), "/queue/subscribe", "您收到了新的消息");
        }
        dao.saveAll(messageSends);
    }

    @Override
    public void sendTemplateMessage(Long userId, Long messageId, Map<String, String> params) {
        List<Long> users = new ArrayList<>();
        users.add(userId);
        this.sendTemplateMessage(users, messageId, params);
    }

    public String changeParams(String v, Map<String, String> params) {
        if (v == null) {
            return "";
        }
        if (params == null) {
            return v;
        }
        for (Map.Entry entry : params.entrySet()) {
            String key = entry.getKey().toString();
            String value = entry.getValue().toString();
            v = v.replaceAll("\\$\\{" + key + "\\}", value);
        }
        return v;
    }
}
