/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.module.system.controller;

import cn.hutool.core.util.StrUtil;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.annotation.Resource;
import jakarta.persistence.EntityManager;
import jakarta.persistence.PersistenceContext;
import jakarta.validation.Valid;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.cache.annotation.CacheConfig;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.data.domain.Page;
import org.springframework.security.crypto.bcrypt.BCryptPasswordEncoder;
import org.springframework.stereotype.Controller;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.bind.annotation.*;
import plus.hiver.common.annotation.PermissionTag;
import plus.hiver.common.api.Result;
import plus.hiver.common.constant.HiverConstant;
import plus.hiver.common.constant.UserConstant;
import plus.hiver.common.dto.RoleDTO;
import plus.hiver.common.entity.Department;
import plus.hiver.common.entity.Role;
import plus.hiver.common.entity.User;
import plus.hiver.common.entity.UserRole;
import plus.hiver.common.redis.RedisTemplateHelper;
import plus.hiver.common.service.*;
import plus.hiver.common.service.mybatis.IUserRoleService;
import plus.hiver.common.utils.NameUtil;
import plus.hiver.common.utils.PageUtil;
import plus.hiver.common.utils.ResultUtil;
import plus.hiver.common.utils.SecurityUtil;
import plus.hiver.common.vo.PageVo;
import plus.hiver.common.vo.SearchVo;
import plus.hiver.module.system.async.AddMessage;

import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 用户管理接口
 *
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
@Slf4j
@RestController
@Controller
@AllArgsConstructor
@Tag(name = "用户接口")
@PermissionTag(permission = "user:*" )
@RequestMapping("/hiver/user")
@CacheConfig(cacheNames = "user")
@Transactional
public class UserController {
    /**
     * 缓存key前缀
     */
    public static final String USER = "username::";

    @Resource
    private UserService userService;

    @Resource
    private RoleService roleService;

    @Resource
    private DepartmentService departmentService;

    @Resource
    private DepartmentHeaderService departmentHeaderService;

    @Resource
    private IUserRoleService iUserRoleService;

    @Resource
    private UserRoleService userRoleService;

    @Resource
    private AddMessage addMessage;

    @Resource
    private SecurityUtil securityUtil;

    @PersistenceContext
    private EntityManager entityManager;

    @Resource
    private RedisTemplateHelper redisTemplateHelper;

    @GetMapping(value = "/info")
    @Operation(summary = "获取当前登录用户接口")
    public Result<User> getUserInfo() {
        User u = securityUtil.getCurrUser();
        // 清除持久上下文环境 避免后面语句导致持久化
        entityManager.detach(u);
        u.setPassword(null);
        return new ResultUtil<User>().setData(u);
    }

    @PostMapping(value = "/unlock")
    @Operation(summary = "解锁验证密码")
    public Result unLock(@RequestParam String password) {
        User u = securityUtil.getCurrUser();
        if (!new BCryptPasswordEncoder().matches(password, u.getPassword())) {
            return ResultUtil.error("密码不正确");
        }
        return ResultUtil.data(null);
    }

    @PostMapping(value = "/edit")
    @Operation(summary = "修改用户自己资料", description = "用户名密码等不会修改 需要username更新缓存")
    @CacheEvict(key = "#u.username")
    public Result editOwn(@RequestBody User u) {
        User old = securityUtil.getCurrUser();
        // 不能修改的字段
        u.setUsername(old.getUsername()).setPassword(old.getPassword()).setType(old.getType()).setStatus(old.getStatus());
        if (u.getDepartmentId() == 0) {
            u.setDepartmentId(null);
        }
        userService.update(u);
        return ResultUtil.success("修改成功");
    }

    /**
     * 线上demo不允许测试账号改密码
     *
     * @param password
     * @param newPass
     * @return
     */
    @PostMapping(value = "/modifyPass")
    @Operation(summary = "修改密码")
    public Result modifyPass(@Parameter(name = "旧密码") @RequestParam String password,
                             @Parameter(name = "新密码") @RequestParam String newPass,
                             @Parameter(name = "密码强度") @RequestParam String passStrength) {
        User user = securityUtil.getCurrUser();
        if (!new BCryptPasswordEncoder().matches(password, user.getPassword())) {
            return ResultUtil.error("旧密码不正确");
        }
        String newEncryptPass = new BCryptPasswordEncoder().encode(newPass);
        user.setPassword(newEncryptPass);
        user.setPassStrength(passStrength);
        userService.update(user);
        // 手动更新缓存
        redisTemplateHelper.delete(USER + user.getUsername());
        return ResultUtil.success("修改密码成功");
    }

    @PostMapping(value = "/changeMobile")
    @Operation(summary = "修改绑定手机")
    public Result changeMobile(@RequestParam String mobile) {
        User u = securityUtil.getCurrUser();
        u.setMobile(mobile);
        userService.update(u);
        // 删除缓存
        redisTemplateHelper.delete(USER + u.getUsername());
        return ResultUtil.success("修改手机号成功");
    }

    @PostMapping(value = "/resetPass")
    @Operation(summary = "重置密码")
    public Result resetPass(@RequestParam Long[] ids) {
        for (Long id : ids) {
            User u = userService.get(id);
            u.setPassword(new BCryptPasswordEncoder().encode("123456"));
            userService.update(u);
            redisTemplateHelper.delete(USER + u.getUsername());
        }
        return ResultUtil.success("操作成功");
    }

    @GetMapping(value = "/getByCondition")
    @Operation(summary = "多条件分页获取用户列表")
    public Result<Page<User>> getByCondition(User user,
                                             SearchVo searchVo,
                                             PageVo pageVo) {
        Page<User> page = userService.findByCondition(user, searchVo, PageUtil.initPage(pageVo));
        for (User u : page.getContent()) {
            // 关联角色
            List<Role> list = iUserRoleService.findByUserId(u.getId());
            List<RoleDTO> roleDTOList = list.stream().map(e -> {
                return new RoleDTO().setId(e.getId()).setName(e.getName()).setDescription(e.getDescription());
            }).collect(Collectors.toList());
            u.setRoles(roleDTOList);
            // 游离态 避免后面语句导致持久化
            entityManager.detach(u);
            u.setPassword(null);
        }
        return new ResultUtil<Page<User>>().setData(page);
    }

    @GetMapping(value = "/getByDepartmentId/{departmentId}")
    @Operation(summary = "多条件分页获取用户列表")
    public Result<List<User>> getByCondition(@PathVariable Long departmentId) {
        List<User> list = userService.findByDepartmentId(departmentId);
        list.forEach(u -> {
            entityManager.detach(u);
            u.setPassword(null);
        });
        return new ResultUtil<List<User>>().setData(list);
    }

    @GetMapping(value = "/searchByName/{username}")
    @Operation(summary = "通过用户名搜索用户")
    public Result<List<User>> searchByName(@PathVariable String username) throws UnsupportedEncodingException {
        List<User> list = userService.findByUsernameLikeAndStatus(URLDecoder.decode(username, "utf-8"), HiverConstant.STATUS_NORMAL);
        list.forEach(u -> {
            entityManager.detach(u);
            u.setPassword(null);
        });
        return new ResultUtil<List<User>>().setData(list);
    }

    @GetMapping(value = "/getAll")
    @Operation(summary = "获取全部用户数据")
    public Result<List<User>> getAll() {
        List<User> list = userService.getAll();
        // 清除持久上下文环境 避免后面语句导致持久化
        list.forEach(u -> {
            entityManager.detach(u);
            u.setPassword(null);
        });
        return new ResultUtil<List<User>>().setData(list);
    }

    @PostMapping(value = "/admin/add")
    @Operation(summary = "添加用户")
    public Result add(@Valid @RequestBody User u,
                      @RequestParam(required = false) Long[] roleIds) {
        // 校验是否已存在
        NameUtil.checkUserInfo(u);
        if (StrUtil.isBlank(u.getNickname())) {
            return ResultUtil.error("nickname不能为空");
        }

        String encryptPass = new BCryptPasswordEncoder().encode(u.getPassword());
        u.setPassword(encryptPass);
        if (u.getDepartmentId() > 0) {
            Department d = departmentService.get(u.getDepartmentId());
            if (d != null) {
                u.setDepartmentTitle(d.getTitle());
            }
        } else {
            u.setDepartmentId(null);
            u.setDepartmentTitle("");
        }

        User user = userService.save(u);

        if (roleIds != null) {
            // 添加角色
            List<UserRole> userRoles = Arrays.asList(roleIds).stream().map(e -> {
                return new UserRole().setUserId(u.getId()).setRoleId(e);
            }).collect(Collectors.toList());
            userRoleService.saveOrUpdateAll(userRoles);
        }
        // 发送创建账号消息
        addMessage.addSendMessage(user.getId());
        return ResultUtil.success("添加成功");
    }

    @PostMapping(value = "/admin/edit")
    @Operation(summary = "管理员修改资料", description = "需要通过id获取原用户信息 需要username更新缓存")
    @CacheEvict(key = "#u.username")
    public Result edit(@RequestBody User u,
                       @RequestParam(required = false) Long[] roleIds) {
        User old = userService.get(u.getId());

        u.setUsername(old.getUsername());
        // 若修改了手机和邮箱判断是否唯一
        if (!old.getMobile().equals(u.getMobile()) && userService.findByMobile(u.getMobile()) != null) {
            return ResultUtil.error("该手机号已绑定其他账户");
        }
        if (!old.getEmail().equals(u.getEmail()) && userService.findByEmail(u.getEmail()) != null) {
            return ResultUtil.error("该邮箱已绑定其他账户");
        }

        if (u.getDepartmentId() > 0) {
            Department d = departmentService.get(u.getDepartmentId());
            if (d != null) {
                u.setDepartmentTitle(d.getTitle());
            }
        } else {
            u.setDepartmentId(null);
            u.setDepartmentTitle("");
        }

        u.setPassword(old.getPassword());
        userService.update(u);
        // 删除该用户角色
        userRoleService.deleteByUserId(u.getId());
        if (roleIds != null) {
            // 新角色
            List<UserRole> userRoles = Arrays.asList(roleIds).stream().map(e -> {
                return new UserRole().setRoleId(e).setUserId(u.getId());
            }).collect(Collectors.toList());
            userRoleService.saveOrUpdateAll(userRoles);
        }
        // 手动删除缓存
        redisTemplateHelper.delete("userRole::" + u.getId());
        redisTemplateHelper.delete("userRole::depIds:" + u.getId());
        redisTemplateHelper.delete("permission::userMenuList:" + u.getId());
        return ResultUtil.success("修改成功");
    }

    @PostMapping(value = "/admin/disable/{userId}")
    @Operation(summary = "后台禁用用户")
    public Result disable(@Parameter(name = "用户唯一id标识") @PathVariable Long userId) {
        User user = userService.get(userId);
        user.setStatus(UserConstant.USER_STATUS_LOCK);
        userService.update(user);
        // 手动更新缓存
        redisTemplateHelper.delete(USER + user.getUsername());
        return ResultUtil.success("操作成功");
    }

    @PostMapping(value = "/admin/enable/{userId}")
    @Operation(summary = "后台启用用户")
    public Result enable(@Parameter(name = "用户唯一id标识") @PathVariable Long userId) {
        User user = userService.get(userId);
        user.setStatus(UserConstant.USER_STATUS_NORMAL);
        userService.update(user);
        // 手动更新缓存
        redisTemplateHelper.delete(USER + user.getUsername());
        return ResultUtil.success("操作成功");
    }

    @PostMapping(value = "/delByIds")
    @Operation(summary = "批量通过ids删除")
    public Result delAllByIds(@RequestParam Long[] ids) {
        for (Long id : ids) {
            User u = userService.get(id);
            // 删除相关缓存
            redisTemplateHelper.delete(USER + u.getUsername());
            redisTemplateHelper.delete("userRole::" + u.getId());
            redisTemplateHelper.delete("userRole::depIds:" + u.getId());
            redisTemplateHelper.delete("permission::userMenuList:" + u.getId());
            redisTemplateHelper.deleteByPattern("department::*");

            userService.delete(id);

            // 删除关联角色
            userRoleService.deleteByUserId(id);
            // 删除关联部门负责人
            departmentHeaderService.deleteByUserId(id);
        }
        return ResultUtil.success("批量通过id删除数据成功");
    }

    @PostMapping(value = "/importData")
    @Operation(summary = "导入用户数据")
    public Result importData(@RequestBody List<User> users) {
        List<Integer> errors = new ArrayList<>();
        List<String> reasons = new ArrayList<>();
        int count = 0;
        for (User u : users) {
            count++;
            // 验证用户名、密码、手机、邮箱不为空
            if (StrUtil.isBlank(u.getUsername()) || StrUtil.isBlank(u.getPassword()) || StrUtil.isBlank(u.getMobile())
                    || StrUtil.isBlank(u.getEmail())) {
                errors.add(count);
                reasons.add("用户名、密码、手机、邮箱不能为空");
                continue;
            }
            // 验证用户名、手机、邮箱唯一
            if (userService.findByUsername(u.getUsername()) != null || userService.findByMobile(u.getMobile()) != null
                    || userService.findByEmail(u.getEmail()) != null) {
                errors.add(count);
                reasons.add("用户名、手机、邮箱已存在");
                continue;
            }
            // 加密密码
            u.setPassword(new BCryptPasswordEncoder().encode(u.getPassword()));
            // 验证部门id正确性
            if (u.getDepartmentId() > 0) {
                Department department = departmentService.findById(u.getDepartmentId());
                if (department == null) {
                    errors.add(count);
                    reasons.add("部门id不存在");
                    continue;
                }
                u.setDepartmentTitle(department.getTitle());
            }
            if (u.getStatus() == null) {
                u.setStatus(UserConstant.USER_STATUS_NORMAL);
            }
            userService.save(u);
            // 分配默认角色
            if (u.getDefaultRole() != null && u.getDefaultRole() == 1) {
                List<Role> roleList = roleService.findByDefaultRole(true);
                if (roleList != null && roleList.size() > 0) {
                    for (Role role : roleList) {
                        UserRole ur = new UserRole().setUserId(u.getId()).setRoleId(role.getId());
                        userRoleService.save(ur);
                    }
                }
            }
        }
        // 批量保存数据
        int successCount = users.size() - errors.size();
        String successMessage = "全部导入成功，共计 " + successCount + " 条数据";
        String failMessage = "导入成功 " + successCount + " 条，失败 " + errors.size() + " 条数据。<br>" +
                "第 " + errors + " 行数据导入出错，错误原因分别为：<br>" + reasons;
        String message;
        if (errors.isEmpty()) {
            message = successMessage;
        } else {
            message = failMessage;
        }
        return ResultUtil.success(message);
    }
}
