/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.module.system.controller;

import cn.hutool.core.util.IdUtil;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.annotation.Resource;
import jakarta.servlet.http.HttpServletResponse;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.web.bind.annotation.*;
import plus.hiver.common.annotation.PermissionTag;
import plus.hiver.common.api.Result;
import plus.hiver.common.redis.RedisTemplateHelper;
import plus.hiver.common.utils.CreateVerifyCode;
import plus.hiver.common.utils.ResultUtil;
import plus.hiver.common.utils.StringUtil;

import java.io.IOException;
import java.util.concurrent.TimeUnit;

/**
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
@Slf4j
@RestController
@AllArgsConstructor
@Tag(name = "验证码接口")
@PermissionTag(permission = "common:captcha:*" )
@RequestMapping("/hiver/common/captcha")
public class CaptchaController {
    @Resource
    private RedisTemplateHelper redisTemplateHelper;

    @GetMapping(value = "/init")
    @Operation(summary = "初始化验证码")
    public Result initCaptcha(@Parameter(name = "仅生成数字") @RequestParam(required = false, defaultValue = "false") Boolean isDigit,
                              @Parameter(name = "验证码长度") @RequestParam(required = false, defaultValue = "4") Integer length) {
        // 生成验证码
        String captchaId = IdUtil.simpleUUID();
        String code;
        if (isDigit) {
            code = StringUtil.randomDigit(length);
        } else {
            code = StringUtil.randomStr(length);
        }
        // 缓存验证码
        redisTemplateHelper.set(captchaId, code, 2L, TimeUnit.MINUTES);
        return ResultUtil.data(captchaId);
    }

    @GetMapping(value = "/draw/{captchaId}")
    @Operation(summary = "根据验证码ID获取图片")
    public void drawCaptcha(@PathVariable("captchaId") String captchaId,
                            HttpServletResponse response) throws IOException {
        // 得到验证码 生成指定验证码
        String code = redisTemplateHelper.get(captchaId);
        CreateVerifyCode vCode = new CreateVerifyCode(116, 36, 4, 10, code);
        response.setContentType("image/png");
        vCode.write(response.getOutputStream());
    }
}
