/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.module.system.controller;

import com.google.gson.Gson;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.annotation.Resource;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.ibatis.annotations.Param;
import org.springframework.data.domain.Page;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.bind.annotation.*;
import plus.hiver.common.annotation.PermissionTag;
import plus.hiver.common.api.Result;
import plus.hiver.common.base.HiverBaseController;
import plus.hiver.common.constant.MessageConstant;
import plus.hiver.common.entity.User;
import plus.hiver.common.service.UserService;
import plus.hiver.common.utils.PageUtil;
import plus.hiver.common.utils.ResultUtil;
import plus.hiver.common.utils.SecurityUtil;
import plus.hiver.common.vo.PageVo;
import plus.hiver.module.system.entity.Message;
import plus.hiver.module.system.entity.MessageSend;
import plus.hiver.module.system.service.MessageSendService;
import plus.hiver.module.system.service.MessageService;

import java.util.HashMap;

/**
 * 消息发送控制器
 *
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
@Slf4j
@RestController
@AllArgsConstructor
@Tag(name = "消息发送管理接口")
@PermissionTag(permission = "messageSend:*" )
@RequestMapping("/hiver/messageSend")
@Transactional
public class MessageSendController extends HiverBaseController<MessageSend, Long> {
    @Resource
    private UserService userService;

    @Resource
    private MessageService messageService;

    @Resource
    private MessageSendService messageSendService;

    @Resource
    private SecurityUtil securityUtil;

    public MessageSendService getService() {
        return messageSendService;
    }

    @GetMapping(value = "/getByCondition")
    @Operation(summary = "多条件分页获取")
    public Result<Page<MessageSend>> getByCondition(MessageSend ms,
                                                    PageVo pv) {
        Page<MessageSend> page = messageSendService.findByCondition(ms, PageUtil.initPage(pv));
        page.getContent().forEach(item -> {
            User u = userService.findById(item.getUserId());
            if (u != null) {
                item.setUsername(u.getUsername());
                item.setNickname(u.getNickname());
            }
            Message m = messageService.findById(item.getMessageId());
            if (m != null) {
                if (m.getIsTemplate()) {
                    Message message = messageSendService.getTemplateMessage(item.getMessageId(),
                            new Gson().fromJson(item.getParams(), HashMap.class));
                    item.setTitle(message.getTitle());
                    item.setContent(message.getContent());
                    item.setType(m.getType());
                } else {
                    item.setTitle(m.getTitle());
                    item.setContent(m.getContent());
                    item.setType(m.getType());
                }
            }
        });
        return new ResultUtil<Page<MessageSend>>().setData(page);
    }

    @GetMapping(value = "/all/{type}")
    @Operation(summary = "多条件分页获取")
    public Result batchOperation(@Param("0全部已读 1全部删除已读") @PathVariable Integer type) {
        User u = securityUtil.getCurrUser();
        if (type == 0) {
            messageSendService.updateStatusByUserId(u.getId(), MessageConstant.MESSAGE_STATUS_READ);
        } else if (type == 1) {
            messageSendService.deleteByUserId(u.getId());
        }
        return ResultUtil.success("操作成功");
    }

    @PostMapping(value = "/edit")
    @Operation(summary = "编辑")
    public Result edit(@RequestBody MessageSend messageSend) {
        if (messageService.findById(messageSend.getMessageId()) != null) {
            messageSendService.update(messageSend);
        }
        return ResultUtil.success("操作成功");
    }
}
