/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.module.system.controller;

import cn.hutool.core.util.StrUtil;
import com.google.gson.Gson;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.annotation.Resource;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.web.bind.annotation.*;
import plus.hiver.common.annotation.PermissionTag;
import plus.hiver.common.api.Result;
import plus.hiver.common.constant.SettingConstant;
import plus.hiver.common.entity.Setting;
import plus.hiver.common.service.SettingService;
import plus.hiver.common.utils.ResultUtil;
import plus.hiver.common.vo.OtherSetting;
import plus.hiver.module.system.vo.EmailSetting;
import plus.hiver.module.system.vo.NoticeSetting;
import plus.hiver.module.system.vo.OssSetting;
import plus.hiver.module.system.vo.SmsSetting;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 基础配置接口
 *
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
@Slf4j
@RestController
@AllArgsConstructor
@Tag(name = "基本配置接口")
@PermissionTag(permission = "setting:*" )
@RequestMapping("/hiver/setting")
public class SettingController {
    @Resource
    private SettingService settingService;

    // 校验配置

    @GetMapping(value = "/seeSecret/{settingName}")
    @Operation(summary = "查看私密配置")
    public Result seeSecret(@PathVariable String settingName) {
        String result = "";
        Setting setting = settingService.findByKey(settingName);
        if (setting == null || StrUtil.isBlank(setting.getValue())) {
            return ResultUtil.error("配置不存在");
        }
        if (settingName.equals(SettingConstant.RUSTFS_OSS)) {
            result = new Gson().fromJson(setting.getValue(), OssSetting.class).getSecretKey();
        } else if (settingName.equals(SettingConstant.ALI_SMS)) {
            result = new Gson().fromJson(setting.getValue(), SmsSetting.class).getSecretKey();
        } else if (settingName.equals(SettingConstant.EMAIL_SETTING)) {
            result = new Gson().fromJson(setting.getValue(), EmailSetting.class).getPassword();
        }
        return ResultUtil.data(result);
    }

    @GetMapping(value = "/oss/check")
    @Operation(summary = "检查OSS配置")
    public Result ossCheck() {
        Setting setting = settingService.findByKey(SettingConstant.OSS_USED);
        if (setting == null || StrUtil.isBlank(setting.getValue())) {
            return ResultUtil.error(501, "您还未配置第三方OSS服务");
        }
        return ResultUtil.data(setting.getValue());
    }

    @GetMapping(value = "/sms/check")
    @Operation(summary = "检查短信配置")
    public Result smsCheck() {
        Setting setting = settingService.findByKey(SettingConstant.SMS_USED);
        if (setting == null || StrUtil.isBlank(setting.getValue())) {
            return ResultUtil.error(501, "您还未配置第三方短信服务");
        }
        return ResultUtil.data(setting.getValue());
    }

    // 查看配置

    @GetMapping(value = "/oss/{serviceName}")
    @Operation(summary = "查看OSS配置")
    public Result<OssSetting> oss(@PathVariable String serviceName) {
        Setting setting = new Setting();
        if (serviceName.equals(SettingConstant.RUSTFS_OSS) || serviceName.equals(SettingConstant.LOCAL_OSS)) {
            setting = settingService.findByKey(SettingConstant.OSS_USED);
        }
        if (setting == null || StrUtil.isBlank(setting.getValue())) {
            return new ResultUtil<OssSetting>().setData(null);
        }
        String value = settingService.findByKey(serviceName).getValue();
        OssSetting ossSetting = new Gson().fromJson(value, OssSetting.class);
        ossSetting.setSecretKey("**********");
        return new ResultUtil<OssSetting>().setData(ossSetting);
    }

    @GetMapping(value = "/sms/{serviceName}")
    @Operation(summary = "查看短信配置")
    public Result<SmsSetting> sms(@PathVariable String serviceName) {
        Setting setting = new Setting();
        if (serviceName.equals(SettingConstant.ALI_SMS)) {
            setting = settingService.findByKey(serviceName);
        }
        if (setting == null || StrUtil.isBlank(setting.getValue())) {
            return new ResultUtil<SmsSetting>().setData(null);
        }
        SmsSetting smsSetting = new Gson().fromJson(setting.getValue(), SmsSetting.class);
        smsSetting.setSecretKey("**********");
        if (smsSetting.getType() != null) {
            Setting code = settingService.findByKey(serviceName + "_" + smsSetting.getType());
            smsSetting.setTemplateCode(code.getValue());
        }
        return new ResultUtil<SmsSetting>().setData(smsSetting);
    }

    @GetMapping(value = "/sms/templateCodes/{serviceName}")
    @Operation(summary = "查看短信模板配置")
    public Result smsTemplateCode(@PathVariable String serviceName) {
        List<Map> list = new ArrayList<>();
        for (SettingConstant.SMS_TYPE SMS_TYPE : SettingConstant.SMS_TYPE.values()) {
            Setting setting = settingService.findByKey(serviceName + "_" + SMS_TYPE.name());
            Map<String, String> map = new HashMap<>();
            String value = "";
            if (StrUtil.isNotBlank(setting.getValue())) {
                value = setting.getValue();
            }
            map.put("type", SMS_TYPE.name());
            map.put("templateCode", value);
            list.add(map);
        }
        return ResultUtil.data(list);
    }

    @GetMapping(value = "/email")
    @Operation(summary = "查看email配置")
    public Result<EmailSetting> email() {
        Setting setting = settingService.findByKey(SettingConstant.EMAIL_SETTING);
        if (setting == null || StrUtil.isBlank(setting.getValue())) {
            return new ResultUtil<EmailSetting>().setData(null);
        }
        EmailSetting emailSetting = new Gson().fromJson(setting.getValue(), EmailSetting.class);
        emailSetting.setPassword("**********");
        return new ResultUtil<EmailSetting>().setData(emailSetting);
    }

    @GetMapping(value = "/other")
    @Operation(summary = "查看其他配置")
    public Result<OtherSetting> other() {
        Setting setting = settingService.findByKey(SettingConstant.OTHER_SETTING);
        if (setting == null || StrUtil.isBlank(setting.getValue())) {
            return new ResultUtil<OtherSetting>().setData(null);
        }
        OtherSetting otherSetting = new Gson().fromJson(setting.getValue(), OtherSetting.class);
        return new ResultUtil<OtherSetting>().setData(otherSetting);
    }

    @GetMapping(value = "/notice")
    @Operation(summary = "查看公告配置")
    public Result<NoticeSetting> notice() {
        Setting setting = settingService.findByKey(SettingConstant.NOTICE_SETTING);
        if (setting == null || StrUtil.isBlank(setting.getValue())) {
            return new ResultUtil<NoticeSetting>().setData(null);
        }
        NoticeSetting noticeSetting = new Gson().fromJson(setting.getValue(), NoticeSetting.class);
        return new ResultUtil<NoticeSetting>().setData(noticeSetting);
    }

    // 配置保存

    @PostMapping(value = "/oss/set")
    @Operation(summary = "OSS配置")
    public Result ossSet(@RequestBody OssSetting ossSetting) {
        String name = ossSetting.getServiceName();
        Setting setting = settingService.findByKey(name);
        if (name.equals(SettingConstant.RUSTFS_OSS)) {
            // 判断是否修改secrectKey 保留原secrectKey 避免保存***加密字符
            if (StrUtil.isNotBlank(setting.getValue()) && !ossSetting.getChanged()) {
                String secrectKey = new Gson().fromJson(setting.getValue(), OssSetting.class).getSecretKey();
                ossSetting.setSecretKey(secrectKey);
            }
        }
        setting.setValue(new Gson().toJson(ossSetting));
        settingService.saveOrUpdate(setting);
        // 保存启用的OSS服务商
        Setting used = settingService.findByKey(SettingConstant.OSS_USED);
        used.setValue(name);
        settingService.saveOrUpdate(used);
        return ResultUtil.data(null);
    }

    @PostMapping(value = "/sms/set")
    @Operation(summary = "短信配置")
    public Result smsSet(@RequestBody SmsSetting smsSetting) {
        String name = smsSetting.getServiceName();
        Setting setting = settingService.findByKey(name);
        if (name.equals(SettingConstant.ALI_SMS)) {
            // 判断是否修改secrectKey 保留原secrectKey 避免保存***加密字符
            if (StrUtil.isNotBlank(setting.getValue()) && !smsSetting.getChanged()) {
                String secrectKey = new Gson().fromJson(setting.getValue(), SmsSetting.class).getSecretKey();
                smsSetting.setSecretKey(secrectKey);
            }
        }
        setting.setValue(new Gson().toJson(smsSetting.setType(null).setTemplateCode(null)));
        settingService.saveOrUpdate(setting);
        // 保存启用的短信服务商
        Setting used = settingService.findByKey(SettingConstant.SMS_USED);
        used.setValue(name);
        settingService.saveOrUpdate(used);
        return ResultUtil.data(null);
    }

    @PostMapping(value = "/sms/templateCode/set")
    @Operation(summary = "短信配置")
    public Result smsTemplateCodeSet(@RequestBody SmsSetting sms) {
        if (StrUtil.isNotBlank(sms.getServiceName()) && StrUtil.isNotBlank(sms.getType())) {
            Setting codeSetting = settingService.findByKey(sms.getServiceName() + "_" + sms.getType());
            codeSetting.setValue(sms.getTemplateCode());
            settingService.saveOrUpdate(codeSetting);
        }
        return ResultUtil.success();
    }

    @PostMapping(value = "/email/set")
    @Operation(summary = "email配置")
    public Result emailSet(@RequestBody EmailSetting emailSetting) {
        Setting setting = settingService.findByKey(SettingConstant.EMAIL_SETTING);
        if (StrUtil.isNotBlank(setting.getValue()) && !emailSetting.getChanged()) {
            String password = new Gson().fromJson(setting.getValue(), EmailSetting.class).getPassword();
            emailSetting.setPassword(password);
        }
        setting.setValue(new Gson().toJson(emailSetting));
        settingService.saveOrUpdate(setting);
        return ResultUtil.data(null);
    }

    @PostMapping(value = "/other/set")
    @Operation(summary = "其他配置")
    public Result otherSet(@RequestBody OtherSetting otherSetting) {
        Setting setting = settingService.findByKey(SettingConstant.OTHER_SETTING);
        setting.setValue(new Gson().toJson(otherSetting));
        settingService.saveOrUpdate(setting);
        return ResultUtil.data(null);
    }

    @PostMapping(value = "/notice/set")
    @Operation(summary = "公告配置")
    public Result noticeSet(@RequestBody NoticeSetting noticeSetting) {
        Setting setting = settingService.findByKey(SettingConstant.NOTICE_SETTING);
        setting.setValue(new Gson().toJson(noticeSetting));
        settingService.saveOrUpdate(setting);
        return ResultUtil.data(null);
    }
}
