/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.module.system.controller;

import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.annotation.Resource;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.Page;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.bind.annotation.*;
import plus.hiver.common.annotation.PermissionTag;
import plus.hiver.common.api.Result;
import plus.hiver.common.entity.StopWord;
import plus.hiver.common.service.StopWordService;
import plus.hiver.common.utils.PageUtil;
import plus.hiver.common.utils.ResultUtil;
import plus.hiver.common.utils.StopWordsUtil;
import plus.hiver.common.vo.PageVo;
import plus.hiver.common.vo.SearchVo;

/**
 * 禁用词管理控制器
 *
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
@Slf4j
@RestController
@Tag(name = "禁用词管理管理接口")
@PermissionTag(permission = "stopWord:*" )
@AllArgsConstructor
@RequestMapping("/hiver/stopWord")
@Transactional
public class StopWordController {
    @Resource
    private StopWordService stopWordService;

    @GetMapping(value = "/getByCondition")
    @Operation(summary = "多条件分页获取")
    public Result<Page<StopWord>> getByCondition(StopWord stopWord, SearchVo searchVo, PageVo pageVo) {
        Page<StopWord> page = stopWordService.findByCondition(stopWord, searchVo, PageUtil.initPage(pageVo));
        return new ResultUtil<Page<StopWord>>().setData(page);
    }

    @PostMapping(value = "/save")
    @Operation(summary = "保存数据")
    public Result<StopWord> save(@RequestBody StopWord stopWord) {
        StopWord s = stopWordService.save(stopWord);
        StopWordsUtil.addWord(stopWord.getTitle());
        return new ResultUtil<StopWord>().setData(s);
    }

    @PostMapping(value = "/edit")
    @Operation(summary = "更新数据")
    public Result<StopWord> update(@RequestBody StopWord stopWord) {
        StopWord old = stopWordService.get(stopWord.getId());
        StopWord s = stopWordService.update(stopWord);
        StopWordsUtil.removeWord(old.getTitle());
        StopWordsUtil.addWord(stopWord.getTitle());
        return new ResultUtil<StopWord>().setData(s);
    }

    @PostMapping(value = "/delByIds")
    @Operation(summary = "批量通过id删除")
    public Result delByIds(@RequestParam Long[] ids) {
        for (Long id : ids) {
            StopWord stopWord = stopWordService.get(id);
            stopWordService.delete(id);
            StopWordsUtil.removeWord(stopWord.getTitle());
        }
        return ResultUtil.success("批量通过id删除数据成功");
    }
}
