/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.module.system.controller;

import cn.hutool.core.util.IdUtil;
import cn.hutool.core.util.StrUtil;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.annotation.Resource;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.web.bind.annotation.*;
import plus.hiver.common.annotation.PermissionTag;
import plus.hiver.common.api.Result;
import plus.hiver.common.constant.HiverConstant;
import plus.hiver.common.constant.MessageConstant;
import plus.hiver.common.constant.SettingConstant;
import plus.hiver.common.redis.RedisTemplateHelper;
import plus.hiver.common.service.UserService;
import plus.hiver.common.sms.SmsUtil;
import plus.hiver.common.utils.CreateVerifyCode;
import plus.hiver.common.utils.IpInfoUtil;
import plus.hiver.common.utils.ResultUtil;
import plus.hiver.common.utils.StringUtil;

import java.io.IOException;
import java.util.concurrent.TimeUnit;

/**
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
@Slf4j
@RestController
@AllArgsConstructor
@Tag(name = "验证码接口")
@PermissionTag(permission = "common:captcha:*" )
@RequestMapping("/hiver/common/captcha")
public class CaptchaController {
    @Resource
    private RedisTemplateHelper redisTemplateHelper;

    @Resource
    private UserService userService;

    @Resource
    private IpInfoUtil ipInfoUtil;

    @Resource
    private SmsUtil smsUtil;

    @GetMapping(value = "/init")
    @Operation(summary = "初始化验证码")
    public Result initCaptcha(@Parameter(name = "仅生成数字") @RequestParam(required = false, defaultValue = "false") Boolean isDigit,
                              @Parameter(name = "验证码长度") @RequestParam(required = false, defaultValue = "4") Integer length) {
        // 生成验证码
        String captchaId = IdUtil.simpleUUID();
        String code;
        if (isDigit) {
            code = StringUtil.randomDigit(length);
        } else {
            code = StringUtil.randomStr(length);
        }
        // 缓存验证码
        redisTemplateHelper.set(captchaId, code, 2L, TimeUnit.MINUTES);
        return ResultUtil.data(captchaId);
    }

    @GetMapping(value = "/draw/{captchaId}")
    @Operation(summary = "根据验证码ID获取图片")
    public void drawCaptcha(@PathVariable("captchaId") String captchaId,
                            HttpServletResponse response) throws IOException {
        // 得到验证码 生成指定验证码
        String code = redisTemplateHelper.get(captchaId);
        CreateVerifyCode vCode = new CreateVerifyCode(116, 36, 4, 10, code);
        response.setContentType("image/png");
        vCode.write(response.getOutputStream());
    }

    @GetMapping(value = "/sendRegistSms/{mobile}")
    @Operation(summary = "发送注册短信验证码")
    public Result sendRegistSmsCode(@PathVariable String mobile, HttpServletRequest request) {
        return sendSms(mobile, MessageConstant.SMS_RANGE_UNREG, SettingConstant.SMS_TYPE.SMS_COMMON.name(), request);
    }

    @GetMapping(value = "/sendLoginSms/{mobile}")
    @Operation(summary = "发送登录短信验证码")
    public Result sendLoginSmsCode(@PathVariable String mobile, HttpServletRequest request) {
        return sendSms(mobile, MessageConstant.SMS_RANGE_REG, SettingConstant.SMS_TYPE.SMS_COMMON.name(), request);
    }

    @GetMapping(value = "/sendResetSms/{mobile}")
    @Operation(summary = "发送重置密码短信验证码")
    public Result sendResetSmsCode(@PathVariable String mobile, HttpServletRequest request) {
        return sendSms(mobile, MessageConstant.SMS_RANGE_REG, SettingConstant.SMS_TYPE.SMS_RESET_PASS.name(), request);
    }

    @GetMapping(value = "/sendEditMobileSms/{mobile}")
    @Operation(summary = "发送修改手机短信验证码")
    public Result sendEditMobileSmsCode(@PathVariable String mobile, HttpServletRequest request) {
        if (userService.findByMobile(mobile) != null) {
            return ResultUtil.error("该手机号已绑定账户");
        }
        return sendSms(mobile, MessageConstant.SMS_RANGE_ALL, SettingConstant.SMS_TYPE.SMS_COMMON.name(), request);
    }

    /**
     * @param mobile       手机号
     * @param range        发送范围 0发送给所有手机号 1只发送给注册手机 2只发送给未注册手机
     * @param templateType 短信模版类型 详见SettingConstant
     */
    public Result sendSms(String mobile, Integer range, String templateType, HttpServletRequest request) {
        if (MessageConstant.SMS_RANGE_REG.equals(range) && userService.findByMobile(mobile) == null) {
            return ResultUtil.error("手机号未注册");
        } else if (MessageConstant.SMS_RANGE_UNREG.equals(range) && userService.findByMobile(mobile) != null) {
            return ResultUtil.error("手机号已注册");
        }
        // IP限流 1分钟限1个请求
        String key = "sendSms:" + ipInfoUtil.getIpAddr(request);
        String value = redisTemplateHelper.get(key);
        if (StrUtil.isNotBlank(value)) {
            return ResultUtil.error("您发送的太频繁啦，请稍后再试");
        }
        // 生成6位数验证码
        String code = StringUtil.randomDigit(6);
        // 缓存验证码
        redisTemplateHelper.set(HiverConstant.PRE_SMS + mobile, code, 5L, TimeUnit.MINUTES);
        // 发送验证码
        smsUtil.sendCode(mobile, code, templateType);
        // 请求成功 标记限流
        redisTemplateHelper.set(key, "sended", 1L, TimeUnit.MINUTES);
        return ResultUtil.success("发送短信验证码成功");
    }
}
