/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.module.system.controller;

import cn.hutool.core.util.StrUtil;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.reflect.TypeToken;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.annotation.Resource;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.cache.annotation.CacheConfig;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.bind.annotation.*;
import plus.hiver.common.annotation.PermissionTag;
import plus.hiver.common.api.Result;
import plus.hiver.common.constant.HiverConstant;
import plus.hiver.common.entity.Department;
import plus.hiver.common.entity.DepartmentHeader;
import plus.hiver.common.entity.User;
import plus.hiver.common.exception.HiverException;
import plus.hiver.common.redis.RedisTemplateHelper;
import plus.hiver.common.service.DepartmentHeaderService;
import plus.hiver.common.service.DepartmentService;
import plus.hiver.common.service.RoleDepartmentService;
import plus.hiver.common.service.UserService;
import plus.hiver.common.utils.CommonUtil;
import plus.hiver.common.utils.HibernateProxyTypeAdapter;
import plus.hiver.common.utils.ResultUtil;
import plus.hiver.common.utils.SecurityUtil;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * 部门管理接口
 *
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
@Slf4j
@RestController
@Tag(name = "部门管理接口")
@PermissionTag(permission = "department:*" )
@AllArgsConstructor
@RequestMapping("/hiver/department")
@CacheConfig(cacheNames = "department")
@Transactional
public class DepartmentController {
    @Resource
    private DepartmentService departmentService;

    @Resource
    private UserService userService;

    @Resource
    private RoleDepartmentService roleDepartmentService;

    @Resource
    private DepartmentHeaderService departmentHeaderService;

    @Resource
    private SecurityUtil securityUtil;

    @Resource
    private RedisTemplateHelper redisTemplateHelper;

    @GetMapping(value = "/getByParentId/{parentId}")
    @Operation(summary = "通过parentId获取")
    public Result<List<Department>> getByParentId(@PathVariable Long parentId,
                                                  @RequestParam(required = false, defaultValue = "true") Boolean openDataFilter) {
        List<Department> list;
        User u = securityUtil.getCurrUser();
        String key = "department::" + parentId + ":" + u.getId() + "_" + openDataFilter;
        String v = redisTemplateHelper.get(key);
        if (StrUtil.isNotBlank(v)) {
            list = new Gson().fromJson(v, new TypeToken<List<Department>>() {
            }.getType());
            return new ResultUtil<List<Department>>().setData(list);
        }
        list = departmentService.findByParentIdOrderBySortOrder(parentId, openDataFilter);
        setInfo(list);
        redisTemplateHelper.set(key, new GsonBuilder().registerTypeAdapterFactory(HibernateProxyTypeAdapter.FACTORY).create().toJson(list), 15L, TimeUnit.DAYS);
        return new ResultUtil<List<Department>>().setData(list);
    }

    @PostMapping(value = "/add")
    @Operation(summary = "添加")
    public Result add(@RequestBody Department department) {
        Department d = departmentService.save(department);
        // 如果不是添加的一级 判断设置上级为父节点标识
        if (!HiverConstant.PARENT_ID.equals(department.getParentId().toString())) {
            Department parent = departmentService.get(department.getParentId());
            if (parent.getIsParent() == null || !parent.getIsParent()) {
                parent.setIsParent(true);
                departmentService.update(parent);
            }
        }
        // 更新缓存
        redisTemplateHelper.deleteByPattern("department::*");
        // 删除数据权限缓存
        redisTemplateHelper.deleteByPattern("userRole::depIds:*");
        return ResultUtil.success("添加成功");
    }

    @PostMapping(value = "/edit")
    @Operation(summary = "编辑")
    public Result edit(@RequestBody Department department,
                       @RequestParam(required = false) Long[] mainHeader,
                       @RequestParam(required = false) Long[] viceHeader) {
        if (department.getId().equals(department.getParentId())) {
            return ResultUtil.error("上级节点不能为自己");
        }
        Department old = departmentService.get(department.getId());
        Long oldParentId = old.getParentId();
        String oldTitle = old.getTitle();
        Department d = departmentService.update(department);
        // 先删除原数据
        departmentHeaderService.deleteByDepartmentId(department.getId());
        List<DepartmentHeader> headers = new ArrayList<>();
        if (mainHeader != null) {
            for (Long id : mainHeader) {
                DepartmentHeader dh = new DepartmentHeader();
                dh.setUserId(id);
                dh.setDepartmentId(d.getId());
                dh.setType(HiverConstant.HEADER_TYPE_MAIN);
                headers.add(dh);
            }
        }
        if (viceHeader != null) {
            for (Long id : viceHeader) {
                DepartmentHeader dh = new DepartmentHeader();
                dh.setUserId(id);
                dh.setDepartmentId(d.getId());
                dh.setType(HiverConstant.HEADER_TYPE_VICE);
                headers.add(dh);
            }
        }
        // 批量保存
        departmentHeaderService.saveOrUpdateAll(headers);
        // 如果该节点不是一级节点 且修改了级别 判断上级还有无子节点
        if (!HiverConstant.PARENT_ID.equals(oldParentId.toString()) && !oldParentId.equals(department.getParentId())) {
            Department parent = departmentService.get(oldParentId);
            List<Department> children = departmentService.findByParentIdOrderBySortOrder(parent.getId(), false);
            if (parent != null && (children == null || children.isEmpty())) {
                parent.setIsParent(false);
                departmentService.update(parent);
            }
        }
        // 若修改了部门名称
        if (!oldTitle.equals(department.getTitle())) {
            userService.updateDepartmentTitle(department.getId(), department.getTitle());
            // 删除所有用户缓存
            redisTemplateHelper.deleteByPattern("user:*");
        }
        // 手动删除所有部门缓存
        redisTemplateHelper.deleteByPattern("department:*");
        return ResultUtil.success("编辑成功");
    }

    @PostMapping(value = "/delByIds")
    @Operation(summary = "批量通过id删除")
    public Result delByIds(@RequestParam Long[] ids) {
        for (Long id : ids) {
            deleteRecursion(id, ids);
        }
        // 手动删除所有部门缓存
        redisTemplateHelper.deleteByPattern("department:*");
        // 删除数据权限缓存
        redisTemplateHelper.deleteByPattern("userRole::depIds:*");
        return ResultUtil.success("批量通过id删除数据成功");
    }

    public void deleteRecursion(Long id, Long[] ids) {
        List<User> list = userService.findByDepartmentId(id);
        if (list != null && list.size() > 0) {
            throw new HiverException("删除失败，包含正被用户使用关联的部门");
        }
        // 获得其父节点
        Department dep = departmentService.get(id);
        Department parent = departmentService.findById(dep.getParentId());
        departmentService.delete(id);
        // 删除关联数据权限
        roleDepartmentService.deleteByDepartmentId(id);
        // 删除关联部门负责人
        departmentHeaderService.deleteByDepartmentId(id);
        // 判断父节点是否还有子节点
        if (parent != null) {
            List<Department> children = departmentService.findByParentIdOrderBySortOrder(parent.getId(), false);
            if (children == null || children.isEmpty()) {
                parent.setIsParent(false);
                departmentService.update(parent);
            }
        }
        // 递归删除
        List<Department> departments = departmentService.findByParentIdOrderBySortOrder(id, false);
        for (Department d : departments) {
            if (!CommonUtil.judgeIds(d.getId(), ids)) {
                deleteRecursion(d.getId(), ids);
            }
        }
    }

    @GetMapping(value = "/search")
    @Operation(summary = "部门名模糊搜索")
    public Result<List<Department>> searchByTitle(@RequestParam String title,
                                                  @RequestParam(required = false, defaultValue = "true") Boolean openDataFilter) {
        List<Department> list = departmentService.findByTitleLikeOrderBySortOrder("%" + title + "%", openDataFilter);
        setInfo(list);
        return new ResultUtil<List<Department>>().setData(list);
    }

    public void setInfo(List<Department> list) {
        // lambda表达式
        list.forEach(item -> {
            if (!HiverConstant.PARENT_ID.equals(item.getParentId().toString())) {
                Department parent = departmentService.get(item.getParentId());
                item.setParentTitle(parent.getTitle());
            } else {
                item.setParentTitle("一级部门");
            }
            // 设置负责人
            item.setMainHeaders(departmentHeaderService.findHeaderByDepartmentId(item.getId(), HiverConstant.HEADER_TYPE_MAIN));
            item.setViceHeaders(departmentHeaderService.findHeaderByDepartmentId(item.getId(), HiverConstant.HEADER_TYPE_VICE));
        });
    }
}
