/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.module.system.controller;

import cn.hutool.http.HtmlUtil;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.annotation.Resource;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.Page;
import org.springframework.messaging.simp.SimpMessagingTemplate;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.bind.annotation.*;
import plus.hiver.common.annotation.PermissionTag;
import plus.hiver.common.api.Result;
import plus.hiver.common.constant.MessageConstant;
import plus.hiver.common.entity.User;
import plus.hiver.common.service.UserService;
import plus.hiver.common.utils.PageUtil;
import plus.hiver.common.utils.ResultUtil;
import plus.hiver.common.vo.PageVo;
import plus.hiver.common.vo.SearchVo;
import plus.hiver.module.system.entity.Message;
import plus.hiver.module.system.entity.MessageSend;
import plus.hiver.module.system.service.MessageSendService;
import plus.hiver.module.system.service.MessageService;

import java.util.ArrayList;
import java.util.List;

/**
 * 消息控制器
 *
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
@Slf4j
@RestController
@AllArgsConstructor
@Tag(name = "消息内容管理接口")
@PermissionTag(permission = "message:*" )
@RequestMapping("/hiver/message")
@Transactional
public class MessageController {
    @Resource
    private MessageService messageService;

    @Resource
    private MessageSendService sendService;

    @Resource
    private UserService userService;

    @Resource
    private SimpMessagingTemplate messagingTemplate;

    @GetMapping(value = "/getByCondition")
    @Operation(summary = "多条件分页获取")
    public Result<Page<Message>> getByCondition(Message message,
                                                SearchVo searchVo,
                                                PageVo pageVo) {
        Page<Message> page = messageService.findByCondition(message, searchVo, PageUtil.initPage(pageVo));
        page.forEach(e -> {
            e.setContentText(HtmlUtil.cleanHtmlTag(e.getContent()));
        });
        return new ResultUtil<Page<Message>>().setData(page);
    }

    @GetMapping(value = "/get/{id}")
    @Operation(summary = "通过id获取")
    public Result<Message> get(@PathVariable Long id) {
        Message message = messageService.get(id);
        message.setContentText(HtmlUtil.filter(message.getContent()));
        return new ResultUtil<Message>().setData(message);
    }

    @PostMapping(value = "/add")
    @Operation(summary = "添加消息")
    public Result addMessage(@RequestBody Message message) {
        Message m = messageService.save(message);
        // 保存消息发送表
        List<MessageSend> messageSends = new ArrayList<>();
        if (MessageConstant.MESSAGE_RANGE_ALL.equals(message.getRange())) {
            // 全体
            List<User> allUser = userService.getAll();
            allUser.forEach(u -> {
                MessageSend ms = new MessageSend();
                ms.setMessageId(m.getId());
                ms.setUserId(u.getId());
                messageSends.add(ms);
            });
            sendService.saveOrUpdateAll(messageSends);
            // 推送
            messagingTemplate.convertAndSend("/topic/subscribe", "您收到了新的系统消息");
        } else if (MessageConstant.MESSAGE_RANGE_USER.equals(message.getRange())) {
            // 指定用户
            for (Long id : message.getUserIds()) {
                MessageSend ms = new MessageSend();
                ms.setMessageId(m.getId());
                ms.setUserId(id);
                messageSends.add(ms);
            }
            sendService.saveOrUpdateAll(messageSends);
            // 推送
            for (Long id : message.getUserIds()) {
                messagingTemplate.convertAndSendToUser(id.toString(), "/queue/subscribe", "您收到了新的消息");
            }
        }
        return ResultUtil.success("添加成功");
    }

    @PostMapping(value = "/edit")
    @Operation(summary = "编辑消息")
    public Result editMessage(@RequestBody Message message) {
        Message m = messageService.update(message);
        return ResultUtil.success("编辑成功");
    }

    @PostMapping(value = "/delByIds")
    @Operation(summary = "删除消息")
    public Result delMessage(@RequestParam Long[] ids) {
        for (Long id : ids) {
            messageService.delete(id);
            // 删除发送表
            sendService.deleteByMessageId(id);
        }
        return ResultUtil.success("删除成功");
    }
}
