/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.module.system.controller;

import cn.hutool.core.util.StrUtil;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.annotation.Resource;
import jakarta.validation.Valid;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.cache.annotation.CacheConfig;
import org.springframework.security.crypto.bcrypt.BCryptPasswordEncoder;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import plus.hiver.common.annotation.*;
import plus.hiver.common.api.Result;
import plus.hiver.common.config.properties.HiverTokenProperties;
import plus.hiver.common.constant.SecurityConstant;
import plus.hiver.common.constant.UserConstant;
import plus.hiver.common.entity.Role;
import plus.hiver.common.entity.User;
import plus.hiver.common.entity.UserRole;
import plus.hiver.common.enums.LogType;
import plus.hiver.common.exception.HiverException;
import plus.hiver.common.redis.RedisTemplateHelper;
import plus.hiver.common.service.RoleService;
import plus.hiver.common.service.UserRoleService;
import plus.hiver.common.service.UserService;
import plus.hiver.common.utils.NameUtil;
import plus.hiver.common.utils.ResultUtil;
import plus.hiver.common.utils.SecurityUtil;
import plus.hiver.module.system.async.AddMessage;

import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * 用户认证相关接口
 *
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
@Slf4j
@RestController
@AllArgsConstructor
@Tag(name = "用户认证相关接口")
@PermissionTag(permission = "auth:*" )
@RequestMapping("/hiver/auth")
@CacheConfig(cacheNames = "user")
@Transactional
public class AuthController {
    public static final String USER = "username::";

    public static final String LOGIN_FAIL_FLAG = "LOGIN_FAIL_FLAG:";

    public static final String LOGIN_TIME_LIMIT = "LOGIN_TIME_LIMIT:";

    public static final Integer LOGIN_FAIL_TIP_TIME = 3;

    @Resource
    private UserService userService;

    @Resource
    private RoleService roleService;

    @Resource
    private UserRoleService userRoleService;

    @Resource
    private AddMessage addMessage;

    @Resource
    private HiverTokenProperties tokenProperties;

    @Resource
    private SecurityUtil securityUtil;

    @Resource
    private RedisTemplateHelper redisTemplateHelper;

    @PostMapping(value = "/login")
    @SystemLog(description = "账号登录", type = LogType.LOGIN)
    @Operation(summary = "账号/手机/邮箱登录")
    public Result login(@RequestParam String username,
                        @RequestParam String password,
                        @RequestParam(required = false) Boolean saveLogin) {
        String loginFailKey = LOGIN_FAIL_FLAG + username;
        String loginTimeKey = LOGIN_TIME_LIMIT + username;

        String valueFailFlag = redisTemplateHelper.get(loginFailKey);
        Long timeRest = redisTemplateHelper.getExpire(loginFailKey, TimeUnit.MINUTES);
        if (StrUtil.isNotBlank(valueFailFlag)) {
            // 超过限制次数
            return ResultUtil.error("登录错误次数超过限制，请" + timeRest + "分钟后再试");
        }
        User user = securityUtil.checkUserPassword(username, password);
        if (user == null) {
            // 记录密码错误次数
            String valueTime = redisTemplateHelper.get(loginTimeKey);
            if (StrUtil.isBlank(valueTime)) {
                valueTime = "0";
            }
            // 获取已登录错误次数
            Integer loginFailTime = Integer.parseInt(valueTime) + 1;
            redisTemplateHelper.set(loginTimeKey, loginFailTime.toString(), tokenProperties.getLoginAfterTime(), TimeUnit.MINUTES);
            if (loginFailTime >= tokenProperties.getLoginTimeLimit()) {
                redisTemplateHelper.set(loginFailKey, "FAIL", tokenProperties.getLoginAfterTime(), TimeUnit.MINUTES);
            }
            int restLoginTime = tokenProperties.getLoginTimeLimit() - loginFailTime;
            if (restLoginTime > 0 && restLoginTime <= LOGIN_FAIL_TIP_TIME) {
                return ResultUtil.error("账号或密码错误，还有" + restLoginTime + "次尝试机会");
            } else if (restLoginTime <= 0) {
                return ResultUtil.error("登录错误次数超过限制，请" + tokenProperties.getLoginAfterTime() + "分钟后再试");
            }
            return ResultUtil.error("账号或密码错误");
        }
        String accessToken = SecurityConstant.TOKEN_SPLIT + securityUtil.getToken(user, saveLogin);
        return ResultUtil.data(accessToken);
    }

    @PostMapping(value = "/smsLogin")
    @SystemLog(description = "短信登录", type = LogType.LOGIN)
    @Operation(summary = "短信登录")
    public Result smsLogin(@RequestParam String mobile,
                           @RequestParam(required = false) Boolean saveLogin) {
        User user = userService.findByMobile(mobile);
        if (user == null) {
            throw new HiverException("手机号不存在");
        }
        String accessToken = securityUtil.getToken(user, saveLogin);
        return ResultUtil.data(accessToken);
    }

    @PostMapping(value = "/register")
    @Operation(summary = "注册用户")
    public Result register(@Valid User u) {
        // 校验是否已存在
        NameUtil.checkUserInfo(u);
        // 默认Nickname
        String nickname = u.getMobile().substring(0, 3) + "****" + u.getMobile().substring(7, 11);
        // 加密密码
        String encryptPass = new BCryptPasswordEncoder().encode(u.getPassword());
        u.setPassword(encryptPass);
        u.setType(UserConstant.USER_TYPE_NORMAL);
        u.setNickname(nickname);
        User user = userService.save(u);
        // 默认角色
        List<Role> roleList = roleService.findByDefaultRole(true);
        if (roleList != null && roleList.size() > 0) {
            for (Role role : roleList) {
                UserRole ur = new UserRole();
                ur.setUserId(user.getId());
                ur.setRoleId(role.getId());
                userRoleService.save(ur);
            }
        }
        // 异步发送创建账号消息
        addMessage.addSendMessage(user.getId());
        return ResultUtil.data(user);
    }

    @PostMapping(value = "/resetByMobile")
    @Operation(summary = "通过短信重置密码")
    public Result resetByMobile(@RequestParam String mobile,
                                @RequestParam String password,
                                @RequestParam String passStrength) {
        User u = userService.findByMobile(mobile);
        String encryptPass = new BCryptPasswordEncoder().encode(password);
        u.setPassword(encryptPass).setPassStrength(passStrength);
        userService.update(u);
        // 删除缓存
        redisTemplateHelper.delete(USER + u.getUsername());
        return ResultUtil.success("重置密码成功");
    }

    @PostMapping(value = "/resetByEmail")
    @Operation(summary = "通过邮箱重置密码")
    public Result resetByEmail(@RequestParam String email,
                               @RequestParam String password,
                               @RequestParam String passStrength) {
        User u = userService.findByEmail(email);
        String encryptPass = new BCryptPasswordEncoder().encode(password);
        u.setPassword(encryptPass);
        u.setPassStrength(passStrength);
        userService.update(u);
        // 删除缓存
        redisTemplateHelper.delete("user::" + u.getUsername());
        return ResultUtil.success("重置密码成功");
    }

    @PostMapping(value = "/logout")
    @Operation(summary = "用户注销")
    public Result logout() {
        try {
            User user = securityUtil.getCurrUser();
            // 删除个人信息
            redisTemplateHelper.delete(USER + user.getUsername());
            // 获得令牌
            String token = redisTemplateHelper.get(SecurityConstant.USER_TOKEN + user.getUsername());
            // 删除在线状态
            redisTemplateHelper.delete(SecurityConstant.ONLINE_USER_PRE + user.getUsername() + ":" + token);
            // 交互token前缀key
            redisTemplateHelper.delete(SecurityConstant.TOKEN_PRE + token);
            // 删除令牌
            redisTemplateHelper.delete(SecurityConstant.USER_TOKEN + user.getUsername());
        } catch (Exception e) {
            log.error("用户注销异常", e);
        }
        return ResultUtil.success("注销成功");
    }
}
