/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.module.system.controller;

import cn.hutool.core.util.StrUtil;
import com.google.gson.Gson;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.annotation.Resource;
import jakarta.servlet.http.HttpServletRequest;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.bind.annotation.*;
import org.thymeleaf.context.Context;
import plus.hiver.common.annotation.PermissionTag;
import plus.hiver.common.api.Result;
import plus.hiver.common.constant.HiverConstant;
import plus.hiver.common.constant.SettingConstant;
import plus.hiver.common.entity.Setting;
import plus.hiver.common.entity.User;
import plus.hiver.common.exception.HiverException;
import plus.hiver.common.redis.RedisTemplateHelper;
import plus.hiver.common.service.SettingService;
import plus.hiver.common.service.UserService;
import plus.hiver.common.utils.*;
import plus.hiver.common.vo.EmailValidate;
import plus.hiver.common.vo.OtherSetting;

import java.util.concurrent.TimeUnit;

/**
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
@Slf4j
@RestController
@AllArgsConstructor
@Tag(name = "邮箱验证接口")
@PermissionTag(permission = "email:*" )
@RequestMapping("/hiver/email")
@Transactional
public class EmailValidateController {
    @Resource
    private EmailUtil emailUtil;

    @Resource
    private RedisTemplateHelper redisTemplateHelper;

    @Resource
    private UserService userService;

    @Resource
    private IpInfoUtil ipInfoUtil;

    @Resource
    private SettingService settingService;

    @Resource
    private SecurityUtil securityUtil;

    public OtherSetting getOtherSetting() {
        Setting setting = settingService.findByKey(SettingConstant.OTHER_SETTING);
        if (StrUtil.isBlank(setting.getValue())) {
            throw new HiverException("系统未配置访问域名，请联系管理员");
        }
        return new Gson().fromJson(setting.getValue(), OtherSetting.class);
    }

    @GetMapping(value = "/sendEditCode/{email}")
    @Operation(summary = "发送修改邮箱验证码")
    public Result sendEditCode(@PathVariable String email,
                               HttpServletRequest request) {
        return sendEmailCode(email, "修改邮箱", "【海文科技】修改邮箱验证", "code-email", request);
    }

    @GetMapping(value = "/sendResetCode/{email}")
    @Operation(summary = "发送重置密码邮箱验证码")
    public Result sendResetCode(@PathVariable String email,
                                HttpServletRequest request) {
        return sendEmailCode(email, "重置密码", "【海文科技】重置密码邮箱验证", "code-email", request);
    }

    /**
     * 发送邮件验证码
     *
     * @param email
     * @param operation
     * @param title
     * @param template
     * @param request
     * @return
     */
    public Result sendEmailCode(String email, String operation, String title, String template, HttpServletRequest request) {
        // 生成验证码 存入相关信息
        EmailValidate e = new EmailValidate();
        e.setOperation(operation);
        // 验证是否注册
        User user = userService.findByEmail(email);
        if ("修改邮箱".equals(operation)) {
            if (user != null) {
                return ResultUtil.error("该邮箱已绑定账号");
            }
            User u = securityUtil.getCurrUserSimple();
            e.setUsername(u.getUsername());
        } else if ("重置密码".equals(operation)) {
            if (user == null) {
                return ResultUtil.error("该邮箱未注册");
            }
            e.setUsername(user.getUsername());
        }

        // IP限流 1分钟限1个请求
        String key = "sendEmailCode:" + ipInfoUtil.getIpAddr(request);
        String value = redisTemplateHelper.get(key);
        if (StrUtil.isNotBlank(value)) {
            return ResultUtil.error("您发送的太频繁啦，请稍后再试");
        }
        String code = StringUtil.randomStr(6);
        e.setCode(code);
        e.setEmail(email);
        e.setFullUrl(getOtherSetting().getDomain());
        Context context = new Context();
        context.setVariables(ObjectUtil.beanToMap(e));
        redisTemplateHelper.set(HiverConstant.PRE_EMAIL + email, new Gson().toJson(e, EmailValidate.class), 10L, TimeUnit.MINUTES);
        emailUtil.sendTemplateMail(email, title, template, context);
        // 请求成功 标记限流
        redisTemplateHelper.set(key, "sended", 1L, TimeUnit.MINUTES);
        return ResultUtil.success("发送成功");
    }

    @PostMapping(value = "/editEmail")
    @Operation(summary = "修改邮箱或重置密码")
    public Result editEmail(@RequestParam String email) {
        User u = securityUtil.getCurrUser();
        u.setEmail(email);
        userService.update(u);
        // 删除缓存
        redisTemplateHelper.delete("user::" + u.getUsername());
        return ResultUtil.success("修改邮箱成功");
    }
}
