/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.module.system.controller;

import cn.hutool.core.util.StrUtil;
import com.google.gson.Gson;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.annotation.Resource;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.bind.annotation.*;
import plus.hiver.common.annotation.PermissionTag;
import plus.hiver.common.api.Result;
import plus.hiver.common.constant.SecurityConstant;
import plus.hiver.common.redis.RedisTemplateHelper;
import plus.hiver.common.utils.IpInfoUtil;
import plus.hiver.common.utils.PageUtil;
import plus.hiver.common.utils.ResultUtil;
import plus.hiver.common.utils.SecurityUtil;
import plus.hiver.common.vo.OnlineUserVo;
import plus.hiver.common.vo.PageVo;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;

/**
 * 在线用户管理控制器
 *
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
@Slf4j
@RestController
@AllArgsConstructor
@Tag(name = "在线用户管理控制器")
@PermissionTag(permission = "onlineUser:*" )
@RequestMapping("/hiver/onlineUser")
@Transactional
public class OnlineUserController {
    @Resource
    private SecurityUtil securityUtil;

    @Resource
    private RedisTemplateHelper redisTemplateHelper;

    @GetMapping(value = "/getAllByPage")
    @Operation(summary = "分页获取全部")
    public Result<Page<OnlineUserVo>> getAllByPage(@RequestParam(required = false) String key,
                                                   PageVo pageVo) {
        Boolean isAdmin = securityUtil.getCurrUser().hasRole("ROLE_ADMIN");
        List<OnlineUserVo> list = new ArrayList<>();
        if (StrUtil.isNotBlank(key)) {
            key = SecurityConstant.ONLINE_USER_PRE + "*" + key + "*";
        } else {
            key = SecurityConstant.ONLINE_USER_PRE + "*";
        }
        Set<String> keys = redisTemplateHelper.scan(key);
        for (String s : keys) {
            String v = redisTemplateHelper.get(s);
            if (StrUtil.isBlank(v)) {
                continue;
            }
            OnlineUserVo userVo = new Gson().fromJson(v, OnlineUserVo.class);
            userVo.setIpInfo(IpInfoUtil.getIpCity(userVo.getIp()));
            // 在线Demo所需 仅管理员可查看令牌 实际可通过权限菜单控制
            if (!isAdmin) {
                userVo.setAccessToken("");
            }
            list.add(userVo);
        }
        Page<OnlineUserVo> page = new PageImpl<OnlineUserVo>(PageUtil.listToPage(pageVo, list), PageUtil.initPage(pageVo), keys.size());
        return new ResultUtil<Page<OnlineUserVo>>().setData(page);
    }

    @PostMapping(value = "/logout")
    @Operation(summary = "批量强制下线")
    public Result logout(@RequestBody List<OnlineUserVo> users) {
        for (OnlineUserVo user : users) {
            if (StrUtil.isBlank(user.getUsername()) || StrUtil.isBlank(user.getAccessToken())) {
                return ResultUtil.error("缺少必要字段");
            }
            // 删除用户记录
            redisTemplateHelper.delete("username::"+user.getUsername());
            if (user.getIsJWT()) {
                // JWT模式
                // 删除在线用户记录
                redisTemplateHelper.delete(SecurityConstant.ONLINE_USER_PRE + user.getUsername());
            } else {
                // redis模式
                redisTemplateHelper.delete(SecurityConstant.TOKEN_PRE + user.getAccessToken());
                redisTemplateHelper.delete(SecurityConstant.USER_TOKEN + user.getUsername());
                // 删除在线用户记录
                redisTemplateHelper.delete(SecurityConstant.ONLINE_USER_PRE + user.getUsername() + ":" + user.getAccessToken());
            }
        }
        return ResultUtil.success("操作成功");
    }
}
