/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.module.system.controller;

import cn.hutool.core.util.StrUtil;
import com.google.gson.Gson;
import com.google.gson.reflect.TypeToken;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.annotation.Resource;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.cache.annotation.CacheConfig;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.bind.annotation.*;
import plus.hiver.common.annotation.PermissionTag;
import plus.hiver.common.api.Result;
import plus.hiver.common.config.security.permission.MySecurityMetadataSource;
import plus.hiver.common.constant.HiverConstant;
import plus.hiver.common.entity.Permission;
import plus.hiver.common.entity.RolePermission;
import plus.hiver.common.entity.User;
import plus.hiver.common.exception.HiverException;
import plus.hiver.common.redis.RedisTemplateHelper;
import plus.hiver.common.service.PermissionService;
import plus.hiver.common.service.RolePermissionService;
import plus.hiver.common.service.mybatis.IPermissionService;
import plus.hiver.common.utils.CommonUtil;
import plus.hiver.common.utils.ResultUtil;
import plus.hiver.common.utils.SecurityUtil;
import plus.hiver.module.system.utils.VoUtil;
import plus.hiver.module.system.vo.MenuVo;

import java.util.Comparator;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * 菜单权限管理接口
 *
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
@Slf4j
@RestController
@AllArgsConstructor
@Tag(name = "菜单权限管理接口")
@PermissionTag(permission = "permission:*" )
@RequestMapping("/hiver/permission")
@CacheConfig(cacheNames = "permission")
@Transactional
public class PermissionController {
    @Resource
    private PermissionService permissionService;

    @Resource
    private RolePermissionService rolePermissionService;

    @Resource
    private IPermissionService iPermissionService;

    @Resource
    private SecurityUtil securityUtil;

    @Resource
    private MySecurityMetadataSource mySecurityMetadataSource;

    @Resource
    private RedisTemplateHelper redisTemplateHelper;

    @GetMapping(value = "/getMenuList")
    @Operation(summary = "获取用户页面菜单数据")
    public Result<List<MenuVo>> getAllMenuList() {
        List<MenuVo> menuList;
        // 读取缓存
        User u = securityUtil.getCurrUser();
        String key = "permission::userMenuList:" + u.getId();
        String v = redisTemplateHelper.get(key);
        if (StrUtil.isNotBlank(v)) {
            menuList = new Gson().fromJson(v, new TypeToken<List<MenuVo>>() {
            }.getType());
            return new ResultUtil<List<MenuVo>>().setData(menuList);
        }

        // 用户所有权限 已排序去重
        List<Permission> list = iPermissionService.findByUserId(u.getId());

        // 筛选0级页面
        menuList = list.stream().filter(p -> HiverConstant.PERMISSION_NAV.equals(p.getType()))
                .sorted(Comparator.comparing(Permission::getSortOrder))
                .map(VoUtil::permissionToMenuVo).collect(Collectors.toList());
        getMenuByRecursion(menuList, list);

        // 缓存
        redisTemplateHelper.set(key, new Gson().toJson(menuList), 15L, TimeUnit.DAYS);
        return new ResultUtil<List<MenuVo>>().setData(menuList);
    }

    private void getMenuByRecursion(List<MenuVo> curr, List<Permission> list) {
        curr.forEach(e -> {
            if (HiverConstant.LEVEL_TWO.equals(e.getLevel())) {
                List<String> buttonPermissions = list.stream()
                        .filter(p -> (e.getId()).equals(p.getParentId()) && HiverConstant.PERMISSION_OPERATION.equals(p.getType()))
                        .sorted(Comparator.comparing(Permission::getSortOrder))
                        .map(Permission::getButtonType).collect(Collectors.toList());
                e.setPermTypes(buttonPermissions);
            } else {
                List<MenuVo> children = list.stream()
                        .filter(p -> (e.getId()).equals(p.getParentId()) && HiverConstant.PERMISSION_PAGE.equals(p.getType()))
                        .sorted(Comparator.comparing(Permission::getSortOrder))
                        .map(VoUtil::permissionToMenuVo).collect(Collectors.toList());
                e.setChildren(children);
                if (e.getLevel() < 3) {
                    getMenuByRecursion(children, list);
                }
            }
        });
    }

    @GetMapping(value = "/getAllList")
    @Operation(summary = "获取权限菜单树")
    public Result<List<Permission>> getAllList() {
        List<Permission> permissionList;
        String key = "permission::allList";
        String v = redisTemplateHelper.get(key);
        if (StrUtil.isNotBlank(v)) {
            permissionList = new Gson().fromJson(v, new TypeToken<List<Permission>>() {
            }.getType());
            return new ResultUtil<List<Permission>>().setData(permissionList);
        }
        List<Permission> list = permissionService.getAll();
        // 0级
        List<Permission> list0 = list.stream().filter(e -> (HiverConstant.LEVEL_ZERO).equals(e.getLevel()))
                .sorted(Comparator.comparing(Permission::getSortOrder)).collect(Collectors.toList());
        getAllByRecursion(list0, list);
        redisTemplateHelper.set(key, new Gson().toJson(list0), 15L, TimeUnit.DAYS);
        return new ResultUtil<List<Permission>>().setData(list0);
    }

    private void getAllByRecursion(List<Permission> curr, List<Permission> list) {
        curr.forEach(e -> {
            List<Permission> children = list.stream().filter(p -> (e.getId()).equals(p.getParentId()))
                    .sorted(Comparator.comparing(Permission::getSortOrder)).collect(Collectors.toList());
            e.setChildren(children);
            setInfo(e);
            if (e.getLevel() < 3) {
                getAllByRecursion(children, list);
            }
        });
    }

    @GetMapping(value = "/getByParentId/{parentId}")
    @Operation(summary = "通过id获取")
    @Cacheable(key = "#parentId")
    public Result<List<Permission>> getByParentId(@PathVariable Long parentId) {
        List<Permission> list = permissionService.findByParentIdOrderBySortOrder(parentId);
        list.forEach(e -> setInfo(e));
        return ResultUtil.data(list);
    }

    @PostMapping(value = "/add")
    @Operation(summary = "添加")
    @CacheEvict(key = "'menuList'")
    public Result<Permission> add(@RequestBody Permission permission) {
        if (permission.getId().equals(permission.getParentId())) {
            return ResultUtil.error("上级节点不能为自己");
        }
        // 判断拦截请求的操作权限按钮名是否已存在
        if (HiverConstant.PERMISSION_OPERATION.equals(permission.getType())) {
            List<Permission> list = permissionService.findByTitle(permission.getTitle());
            if (list != null && list.size() > 0) {
                return new ResultUtil<Permission>().setErrorMsg("名称已存在");
            }
        }
        // 如果不是添加的一级 判断设置上级为父节点标识
        if (!HiverConstant.PARENT_ID.equals(permission.getParentId().toString())) {
            Permission parent = permissionService.get(permission.getParentId());
            if (parent.getIsParent() == null || !parent.getIsParent()) {
                parent.setIsParent(true);
                permissionService.update(parent);
            }
        }
        Permission u = permissionService.save(permission);
        // 重新加载权限
        mySecurityMetadataSource.loadResourceDefine();
        // 手动删除缓存
        redisTemplateHelper.deleteByPattern("permission:*");
        return new ResultUtil<Permission>().setData(u);
    }

    @PostMapping(value = "/edit")
    @Operation(summary = "编辑")
    public Result<Permission> edit(@RequestBody Permission permission) {
        if (permission.getId().equals(permission.getParentId())) {
            return ResultUtil.error("上级节点不能为自己");
        }
        // 判断拦截请求的操作权限按钮名是否已存在
        if (HiverConstant.PERMISSION_OPERATION.equals(permission.getType())) {
            // 若名称修改
            Permission p = permissionService.get(permission.getId());
            if (!p.getTitle().equals(permission.getTitle())) {
                List<Permission> list = permissionService.findByTitle(permission.getTitle());
                if (list != null && list.size() > 0) {
                    return ResultUtil.error("名称已存在");
                }
            }
        }
        Permission old = permissionService.get(permission.getId());
        Long oldParentId = old.getParentId();
        Permission u = permissionService.update(permission);
        // 如果该节点不是一级节点 且修改了级别 判断上级还有无子节点
        if (!HiverConstant.PARENT_ID.equals(oldParentId.toString()) && !oldParentId.equals(permission.getParentId())) {
            Permission parent = permissionService.get(oldParentId);
            List<Permission> children = permissionService.findByParentIdOrderBySortOrder(parent.getId());
            if (parent != null && (children == null || children.isEmpty())) {
                parent.setIsParent(false);
                permissionService.update(parent);
            }
        }
        // 重新加载权限
        mySecurityMetadataSource.loadResourceDefine();
        // 手动批量删除缓存
        redisTemplateHelper.deleteByPattern("user:*");
        redisTemplateHelper.deleteByPattern("permission:*");
        return ResultUtil.data(u);
    }

    @PostMapping(value = "/delByIds")
    @Operation(summary = "批量通过id删除")
    @CacheEvict(key = "'menuList'")
    public Result delByIds(@RequestParam Long[] ids) {
        for (Long id : ids) {
            deleteRecursion(id, ids);
        }
        // 重新加载权限
        mySecurityMetadataSource.loadResourceDefine();
        // 手动删除缓存
        redisTemplateHelper.deleteByPattern("permission:*");
        return ResultUtil.success("批量通过id删除数据成功");
    }

    public void deleteRecursion(Long id, Long[] ids) {
        List<RolePermission> list = rolePermissionService.findByPermissionId(id);
        if (list != null && list.size() > 0) {
            throw new HiverException("删除失败，包含正被用户使用关联的菜单");
        }
        // 获得其父节点
        Permission p = permissionService.get(id);
        Permission parent = permissionService.findById(p.getParentId());
        permissionService.delete(id);
        // 判断父节点是否还有子节点
        if (parent != null) {
            List<Permission> children = permissionService.findByParentIdOrderBySortOrder(parent.getId());
            if (children == null || children.isEmpty()) {
                parent.setIsParent(false);
                permissionService.update(parent);
            }
        }
        // 递归删除
        List<Permission> permissions = permissionService.findByParentIdOrderBySortOrder(id);
        for (Permission pe : permissions) {
            if (!CommonUtil.judgeIds(pe.getId(), ids)) {
                deleteRecursion(pe.getId(), ids);
            }
        }
    }

    @GetMapping(value = "/search")
    @Operation(summary = "搜索菜单")
    public Result<List<Permission>> searchPermissionList(@RequestParam String title) {
        List<Permission> list = permissionService.findByTitleLikeOrderBySortOrder("%" + title + "%");
        list.forEach(e -> setInfo(e));
        return new ResultUtil<List<Permission>>().setData(list);
    }

    public void setInfo(Permission permission) {
        if (!HiverConstant.PARENT_ID.equals(permission.getParentId().toString())) {
            Permission parent = permissionService.get(permission.getParentId());
            permission.setParentTitle(parent.getTitle());
        } else {
            permission.setParentTitle("一级菜单");
        }
    }
}
