/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.module.system.controller;

import cn.hutool.core.date.DateUtil;
import cn.hutool.core.util.StrUtil;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.annotation.Resource;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.bind.annotation.*;
import plus.hiver.common.annotation.PermissionTag;
import plus.hiver.common.api.Result;
import plus.hiver.common.redis.RedisTemplateHelper;
import plus.hiver.common.utils.PageUtil;
import plus.hiver.common.utils.ResultUtil;
import plus.hiver.common.vo.PageVo;
import plus.hiver.module.system.vo.RedisInfo;
import plus.hiver.module.system.vo.RedisVo;

import java.util.*;
import java.util.concurrent.TimeUnit;

/**
 * Redis缓存管理接口
 *
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
@Slf4j
@RestController
@AllArgsConstructor
@Tag(name = "Redis缓存管理接口")
@PermissionTag(permission = "redis:*" )
@RequestMapping("/hiver/redis")
@Transactional
public class RedisController {
    /**
     * 最大获取10万个键值
     */
    private static final int maxSize = 100000;

    @Resource
    private RedisTemplateHelper redisTemplateHelper;

    @GetMapping(value = "/getAllByPage")
    @Operation(summary = "分页获取全部")
    public Result<Page<RedisVo>> getAllByPage(@RequestParam(required = false) String key,
                                              PageVo pageVo) {
        List<RedisVo> list = new ArrayList<>();
        if (StrUtil.isNotBlank(key)) {
            key = "*" + key + "*";
        } else {
            key = "*";
        }
        Set<String> keys = redisTemplateHelper.scan(key);
        int size = keys.size();
        // 限制10万个
        if (size > maxSize) {
            size = maxSize;
        }
        int i = 0;
        for (String s : keys) {
            if (i > size) {
                break;
            }
            RedisVo redisVo = new RedisVo(s, "", null);
            list.add(redisVo);
            i++;
        }
        Page<RedisVo> page = new PageImpl<RedisVo>(PageUtil.listToPage(pageVo, list), PageUtil.initPage(pageVo), size);
        page.getContent().forEach(e -> {
            String value = "";
            try {
                value = redisTemplateHelper.get(e.getKey());
                if (value.length() > 150) {
                    value = value.substring(0, 150) + "...";
                }
            } catch (Exception exception) {
                value = "非字符格式数据";
            }
            e.setValue(value);
            e.setExpireTime(redisTemplateHelper.getExpire(e.getKey(), TimeUnit.SECONDS));
        });
        return new ResultUtil<Page<RedisVo>>().setData(page);
    }

    @GetMapping(value = "/getByKey/{key}")
    @Operation(summary = "通过key获取")
    public Result getByKey(@PathVariable String key) {
        Map<String, Object> map = new HashMap<>();
        String value = redisTemplateHelper.get(key);
        Long expireTime = redisTemplateHelper.getExpire(key, TimeUnit.SECONDS);
        map.put("value", value);
        map.put("expireTime", expireTime);
        return ResultUtil.data(map);
    }

    @PostMapping(value = "/save")
    @Operation(summary = "添加或编辑")
    public Result save(@RequestParam String key,
                       @RequestParam String value,
                       @RequestParam Long expireTime) {
        if (expireTime < 0) {
            redisTemplateHelper.set(key, value);
        } else if (expireTime > 0) {
            redisTemplateHelper.set(key, value, expireTime, TimeUnit.SECONDS);
        }
        return ResultUtil.success("保存成功");
    }

    @PostMapping(value = "/delByKeys")
    @Operation(summary = "批量删除")
    public Result delByKeys(@RequestParam String[] keys) {
        for (String key : keys) {
            redisTemplateHelper.delete(key);
        }
        return ResultUtil.success("删除成功");
    }

    @PostMapping(value = "/delAll")
    @Operation(summary = "全部删除")
    public Result delAll() {
        redisTemplateHelper.deleteByPattern("*");
        return ResultUtil.success("删除成功");
    }

    @GetMapping(value = "/getKeySize")
    @Operation(summary = "获取实时key大小")
    public Result getKeySize() {
        Map<String, Object> map = new HashMap<>(16);
        map.put("keySize", redisTemplateHelper.getConnectionFactory().getConnection().dbSize());
        map.put("time", DateUtil.format(new Date(), "HH:mm:ss"));
        return ResultUtil.data(map);
    }

    @GetMapping(value = "/getMemory")
    @Operation(summary = "获取实时内存大小")
    public Result getMemory() {
        Map<String, Object> map = new HashMap<>(16);
        Properties memory = redisTemplateHelper.getConnectionFactory().getConnection().info("memory");
        map.put("memory", memory.get("used_memory"));
        map.put("time", DateUtil.format(new Date(), "HH:mm:ss"));
        return ResultUtil.data(map);
    }

    @GetMapping(value = "/info")
    @Operation(summary = "获取Redis信息")
    public Result info() {
        List<RedisInfo> infoList = new ArrayList<>();
        Properties properties = redisTemplateHelper.getConnectionFactory().getConnection().info();
        Set<Object> keys = properties.keySet();
        for (Object key : keys) {
            String value = properties.get(key).toString();
            RedisInfo redisInfo = new RedisInfo();
            redisInfo.setKey(key.toString());
            redisInfo.setValue(value);
            infoList.add(redisInfo);
        }
        return ResultUtil.data(infoList);
    }
}
