/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.module.system.controller;

import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.annotation.Resource;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.Page;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;
import plus.hiver.common.annotation.PermissionTag;
import plus.hiver.common.api.Result;
import plus.hiver.common.base.HiverBaseController;
import plus.hiver.common.base.HiverBaseService;
import plus.hiver.common.constant.HiverConstant;
import plus.hiver.common.entity.Tenant;
import plus.hiver.common.service.TenantService;
import plus.hiver.common.utils.PageUtil;
import plus.hiver.common.utils.ResultUtil;
import plus.hiver.common.vo.PageVo;
import plus.hiver.common.vo.SearchVo;

import java.util.List;

/**
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
@Slf4j
@RestController
@AllArgsConstructor
@Tag(name = "租户管理接口")
@PermissionTag(permission = "tenant:*" )
@RequestMapping("/hiver/tenant")
@Transactional
public class TenantController extends HiverBaseController<Tenant, Long> {
    @Resource
    private TenantService tenantService;

    @Override
    public HiverBaseService<Tenant, Long> getService() {
        return tenantService;
    }

    @GetMapping(value = "/getByCondition")
    @Operation(summary = "多条件分页获取")
    public Result<Page<Tenant>> getByCondition(Tenant tenant, SearchVo searchVo, PageVo pageVo) {
        Page<Tenant> page = tenantService.findByCondition(tenant, searchVo, PageUtil.initPage(pageVo));
        return new ResultUtil<Page<Tenant>>().setData(page);
    }

    @GetMapping(value = "/getTenant/{name}")
    @Operation(summary = "通过名称获取租户数据")
    public Result<Tenant> getTenant(@PathVariable String name) {
        Tenant tenant = tenantService.findByName(name);
        return new ResultUtil<Tenant>().setData(tenant);
    }

    @GetMapping(value = "/getAll")
    @Operation(summary = "获取全部租户数据")
    public Result<List<Tenant>> getAll() {
        List<Tenant> list = tenantService.findByStatusAndDeleted(HiverConstant.STATUS_NORMAL, 0);
        return new ResultUtil<List<Tenant>>().setData(list);
    }
}
