/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.module.system.serviceimpl;

import cn.hutool.core.date.DateUtil;
import cn.hutool.core.util.StrUtil;
import jakarta.annotation.Nullable;
import jakarta.persistence.criteria.*;
import jakarta.transaction.Transactional;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.stereotype.Service;
import plus.hiver.common.base.HiverBaseServiceImpl;
import plus.hiver.common.vo.SearchVo;
import plus.hiver.module.system.dao.MessageDao;
import plus.hiver.module.system.entity.Message;
import plus.hiver.module.system.service.MessageService;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

/**
 * 消息内容接口实现
 *
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
@Slf4j
@Service
@Transactional
public class MessageServiceImpl extends HiverBaseServiceImpl<Message, Long, MessageDao> implements MessageService {
    public MessageServiceImpl(MessageDao dao) {
        super(dao);
    }

    @Override
    public Page<Message> findByCondition(Message message, SearchVo searchVo, Pageable pageable) {
        return dao.findAll(new Specification<Message>() {
            @Nullable
            @Override
            public Predicate toPredicate(Root<Message> root, CriteriaQuery<?> cq, CriteriaBuilder cb) {
                Path<String> titleField = root.get("title");
                Path<String> contentField = root.get("content");
                Path<String> typeField = root.get("type");
                Path<Boolean> isTemplateField = root.get("isTemplate");
                Path<Date> createTimeField = root.get("createTime");
                List<Predicate> list = new ArrayList<>();
                // 模糊搜素
                if (StrUtil.isNotBlank(message.getTitle())) {
                    list.add(cb.like(titleField, '%' + message.getTitle() + '%'));
                }
                if (StrUtil.isNotBlank(message.getContent())) {
                    list.add(cb.like(contentField, '%' + message.getContent() + '%'));
                }
                if (message.getType() != null) {
                    list.add(cb.equal(typeField, message.getType()));
                }
                if (message.getIsTemplate() != null) {
                    list.add(cb.equal(isTemplateField, message.getIsTemplate()));
                }
                // 创建时间
                if (StrUtil.isNotBlank(searchVo.getStartDate()) && StrUtil.isNotBlank(searchVo.getEndDate())) {
                    Date start = DateUtil.parse(searchVo.getStartDate());
                    Date end = DateUtil.parse(searchVo.getEndDate());
                    list.add(cb.between(createTimeField, start, DateUtil.endOfDay(end)));
                }
                Predicate[] arr = new Predicate[list.size()];
                cq.where(list.toArray(arr));
                return null;
            }
        }, pageable);
    }

    @Override
    public List<Message> findByCreateSend(Boolean createSend) {
        return dao.findByCreateSend(createSend);
    }
}
