/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.module.system.controller;

import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.annotation.Resource;
import jakarta.validation.Valid;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.Page;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.bind.annotation.*;
import plus.hiver.common.annotation.PermissionTag;
import plus.hiver.common.api.Result;
import plus.hiver.common.constant.HiverConstant;
import plus.hiver.common.entity.Tenant;
import plus.hiver.common.service.TenantService;
import plus.hiver.common.service.UserService;
import plus.hiver.common.utils.PageUtil;
import plus.hiver.common.utils.ResultUtil;
import plus.hiver.common.utils.SecurityUtil;
import plus.hiver.common.vo.PageVo;
import plus.hiver.common.vo.SearchVo;

import java.util.List;

/**
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
@Slf4j
@RestController
@AllArgsConstructor
@Tag(name = "租户管理接口")
@PermissionTag(permission = "tenant:*" )
@RequestMapping("/hiver/tenant")
@Transactional
public class TenantController {
    @Resource
    private TenantService tenantService;

    @Resource
    private UserService userService;

    @Resource
    private SecurityUtil securityUtil;

    @GetMapping(value = "/getByCondition")
    @Operation(summary = "多条件分页获取")
    public Result<Page<Tenant>> getByCondition(Tenant tenant, SearchVo searchVo, PageVo pageVo) {
        Page<Tenant> page = tenantService.findByCondition(tenant, searchVo, PageUtil.initPage(pageVo));
        return new ResultUtil<Page<Tenant>>().setData(page);
    }

    @PostMapping(value = "/add")
    @Operation(summary = "添加租户")
    public Result add(@Valid @RequestBody Tenant tenant) {
        // 校验是否已存在
        if (tenantService.findByName(tenant.getName()) != null) {
            return ResultUtil.error("租户已存在");
        }
        tenantService.save(tenant);
        return ResultUtil.success("添加成功");
    }

    @PostMapping(value = "/edit")
    @Operation(summary = "编辑租户")
    public Result edit(@Valid @RequestBody Tenant tanant) {
        tenantService.update(tanant);
        return ResultUtil.success("修改成功");
    }

    @PostMapping(value = "/status")
    @Operation(summary = "修改租户状态")
    public Result disable(@RequestParam Long id, @RequestParam Boolean enable) {
        Tenant tenant = tenantService.get(id);
        if (enable) {
            tenant.setStatus(HiverConstant.STATUS_NORMAL);
        } else {
            tenant.setStatus(HiverConstant.STATUS_DISABLE);
        }
        tenantService.update(tenant);
        return ResultUtil.success("操作成功");
    }

    @PostMapping(value = "/delByIds")
    @Operation(summary = "批量通过ids删除")
    public Result delAllByIds(@RequestParam Long[] ids) {
        for (Long id : ids) {
            Long count = userService.countByTenantId(id);
            if (count > 0) {
                return ResultUtil.error("该租户下有用户，请先删除用户");
            }
            if (id.equals(HiverConstant.DEFAULT_TENANT_ID)) {
                return ResultUtil.error("系统租户不允许删除");
            }
            tenantService.delete(id);
        }
        return ResultUtil.success("批量通过id删除数据成功");
    }

    @GetMapping(value = "/info")
    @Operation(summary = "获得租户信息")
    public Result<Tenant> info() {
        Long tenantId = securityUtil.getCurrUser().getTenantId();
        Tenant tenant = tenantService.get(tenantId);
        return ResultUtil.data(tenant);
    }

    @GetMapping(value = "/getAll")
    @Operation(summary = "获取全部租户数据")
    public Result<List<Tenant>> getAll() {
        List<Tenant> list = tenantService.findByStatusAndDeleted(HiverConstant.STATUS_NORMAL, 0);
        return new ResultUtil<List<Tenant>>().setData(list);
    }
}
