/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.tess4j.service;

import net.sourceforge.tess4j.ITesseract;
import net.sourceforge.tess4j.Tesseract;
import net.sourceforge.tess4j.TesseractException;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.web.multipart.MultipartFile;

import java.io.File;

/**
 * OCR服务
 *
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
@Service
public class OCRService {
    private final ITesseract tesseract;

    public OCRService(@Value("${tesseract.datapath}") String tessDataPath) {
        this.tesseract = new Tesseract();
        // 设置 tessdata 目录（必须包含 chi_sim.traineddata、eng.traineddata 等）
        this.tesseract.setDatapath(tessDataPath);
        // 默认语言可以在方法中动态指定，此处不设置
    }

    /**
     * 对上传的图片文件进行 OCR 识别，返回识别出的文字
     *
     * @param file 上传的图片文件（MultipartFile）
     * @return 识别出的文字内容
     * @throws Exception 如果识别失败或文件有问题，抛出异常
     */
    public String doOCR(MultipartFile file, String language) throws Exception {
        // 1. MultipartFile 转临时 File
        File tempImageFile = convertMultipartFileToFile(file);
        if (!tempImageFile.exists()) {
            throw new IllegalArgumentException("图片文件无效或转换失败");
        }
        try {
            // 2. 设置识别语言（如 chi_sim, eng）
            tesseract.setLanguage(language);
            // 3. 执行 OCR
            String result = tesseract.doOCR(tempImageFile);
            // 4. 删除临时文件
            tempImageFile.delete();
            return result;
        } catch (TesseractException e) {
            throw new Exception("Tesseract 引擎错误: " + e.getMessage(), e);
        }
    }

    /**
     * 将 MultipartFile 转换为临时 File（用于 Tesseract 识别）
     *
     * @param multipartFile 上传的文件
     * @return 临时 File 对象
     * @throws Exception 转换失败时抛出异常
     */
    private File convertMultipartFileToFile(org.springframework.web.multipart.MultipartFile multipartFile) throws Exception {
        if (multipartFile == null || multipartFile.isEmpty()) {
            throw new IllegalArgumentException("上传的文件为空");
        }
        // 创建临时文件，后缀根据原文件类型，或统一用 png
        File tempFile = File.createTempFile("ocr_", ".png");
        multipartFile.transferTo(tempFile);
        return tempFile;
    }
}
