package plus.ibatis.hbatis.orm;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.ParameterizedType;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

import org.apache.ibatis.binding.MapperRegistry;
import org.apache.ibatis.builder.SqlSourceBuilder;
import org.apache.ibatis.mapping.MappedStatement;
import org.apache.ibatis.mapping.MappedStatement.Builder;
import org.apache.ibatis.mapping.ResultMap;
import org.apache.ibatis.mapping.ResultMapping;
import org.apache.ibatis.mapping.SqlCommandType;
import org.apache.ibatis.parsing.XNode;
import org.apache.ibatis.session.Configuration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import plus.ibatis.hbatis.core.metaDescriber.EntityClassDescriber;
import plus.ibatis.hbatis.core.util.EntityClassDescriberHelper;
import plus.ibatis.hbatis.orm.annotation.EntityResultMapping;
import plus.ibatis.hbatis.orm.sql.KeyGeneratorBuilder;
import plus.ibatis.hbatis.orm.sql.ResultMapsBuilder;
import plus.ibatis.hbatis.orm.sql.SqlBuilder;
import plus.ibatis.hbatis.orm.util.XNodeUtil;

/**
 * HbatisStatementBuilder
 * @author zz
 * @version 1.0.0
 * @since 1.0.0
 */
public class HbatisStatementBuilder {

	private static final Logger logger = LoggerFactory.getLogger(HbatisStatementBuilder.class);

	private SqlSourceBuilder sqlSourceBuilder;

	private Set<Method> baseMethodSet = new HashSet<Method>();

	/**
	 * 数据库类型
	 */
	private String dialect = "mysql";

	private Collection<Class<?>> mappers;

	private Configuration configuration;

	private Class<?> baseMapperClass = null;
	static Map<Configuration,HbatisStatementBuilder> factory = new ConcurrentHashMap<>();
	public static synchronized HbatisStatementBuilder getInstance(Configuration configuration) {
		if(factory.containsKey(configuration)) {
			return factory.get(configuration);
		}
		HbatisStatementBuilder b = new HbatisStatementBuilder(configuration);
		factory.put(configuration, b);
		return b;
	}
	private KeyGeneratorBuilder keyGeneratorBuilder;
	protected HbatisStatementBuilder( Configuration configuration) {
		this(configuration,HBatisConfiguration.getBaseMapper());
	}
	@SuppressWarnings("rawtypes")
	protected HbatisStatementBuilder( Configuration configuration,Class<?> baseMapperClass) {
		this.configuration = configuration;
		this.baseMapperClass = baseMapperClass;
		try {
			Method[] methods = baseMapperClass.getMethods();
			Collections.addAll(baseMethodSet, methods);
			logger.info("BaseMapper({}) statements:{}", baseMapperClass, baseMethodSet);
			addMapperIfNotExists(baseMapperClass);
			Class[] superClas = baseMapperClass.getInterfaces();
			for(Class superCla:superClas) {
				addMapperIfNotExists(superCla);
			}
			addMapperIfNotExists(baseMapperClass);
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
		MapperRegistry r = configuration.getMapperRegistry();

		this.mappers = r.getMappers();
		sqlSourceBuilder = new SqlSourceBuilder(configuration);
		// key generator
		Class<KeyGeneratorBuilder> keyBuilderClass;
		try {
			keyBuilderClass = HBatisConfiguration.getKeyGeneratorBuilder(this.dialect);
			this.keyGeneratorBuilder = keyBuilderClass.newInstance();
			
		} catch (ClassNotFoundException | InstantiationException | IllegalAccessException e) {
			throw new RuntimeException(e);
		}
	}
	private void addMapperIfNotExists(Class<?> mapperClass) {
		if (!configuration.hasMapper(mapperClass)) {
			configuration.addMapper(mapperClass);
		}
	}
	public void build() throws ClassNotFoundException, InstantiationException, IllegalAccessException {
		if (this.mappers == null || this.mappers.isEmpty())
			return;
		
		this.initDao();
	}

	private boolean hasProccedMapperClass(Class<?> mapperClass) {
		String resultMapId = mapperClass.getName() +"."+HbatisConstants.ENTITY_RESULTMAP_NAME;
		return configuration.hasResultMap(resultMapId);
	}
	public void addMapper(Class<?> mapperClass) {
		if(hasProccedMapperClass(mapperClass)) {
			logger.debug("Mapper interface has registed."+mapperClass);
			return ;
		}
		this.processMapper(mapperClass);
	}
	private void processMapper(Class<?> mapperClass) {
		if (baseMapperClass.isAssignableFrom(mapperClass) && !baseMapperClass.equals(mapperClass)) {
			logger.info("Preprocessing mapper [BaseResultMap / BaseColumnList]:{}", mapperClass);
			EntityClassDescriber<?> entityDescriber = getEntityClassDescriber(mapperClass);
			
			addResultMaps(mapperClass,entityDescriber);
			addBaseColumns(mapperClass,entityDescriber);
			for (Method m : baseMethodSet) {
				addStatement(mapperClass, m,entityDescriber);
			}
		}
	}
	protected void initDao() {
		for (Class<?> mapperClass : mappers) {
			this.processMapper(mapperClass);
		}
	}

	

	public SqlSourceBuilder getSqlSourceBuilder() {
		return sqlSourceBuilder;
	}
	public String getDialect() {
		return dialect;
	}
	private static Class<?> getEntityClassByInterface(Class<?> mapperClass) {
		ParameterizedType pt = (ParameterizedType) mapperClass.getGenericInterfaces()[0];

		Class<?> entityClass = (Class<?>) (pt.getActualTypeArguments()[0]);
		return entityClass;
	}

	private Method getMapperMethod(Class<?> mapperClass, Method baseMethod) {
		Method[] methods = mapperClass.getMethods();
		for (int i = 0, len = methods.length; i < len; i++) {
			Method m = methods[i];
			if (m.getName().equals(baseMethod.getName())) {
				return m;
			}
		}
		throw new RuntimeException("method not found");
	}
	/**
	 * 获取基类声明
	 * @param baseMapperCla
	 * @param mapperMethod
	 * @return
	 */
	@SuppressWarnings("rawtypes")
	private String getBaseStatementId(Class<?> baseMapperCla,Method mapperMethod) {
		String baseStatementId = baseMapperCla.getName() + "." + mapperMethod.getName();
		if (configuration.hasStatement(baseStatementId,false)) {
			return baseStatementId;
		} else {
			Class[] superClaList = baseMapperCla.getInterfaces();
			for(Class superCla:superClaList) {
				String tmp = getBaseStatementId(superCla,mapperMethod);
				if(tmp != null) {
					return tmp;
				}
			}
			return null;
		}
	}
	
	private synchronized String addStatement(Class<?> mapperClass, Method m,EntityClassDescriber<?> entityDescriber) {
		Method mapperMethod = this.getMapperMethod(mapperClass, m);
		Class<?> entityClass = entityDescriber.getEntityClass();
		String methodName = mapperMethod.getName();
		// 定义子mapper的statement1
		String statementId = mapperClass.getName() + "." + methodName;
		logger.debug("statement id:{}", statementId);
		if (configuration.hasStatement(statementId,false)) {
			return statementId;
		}
		String baseStatementId = this.getBaseStatementId(this.baseMapperClass, mapperMethod);
		logger.debug("statement id:{},baseStatementId:{}", statementId, baseStatementId);
		MappedStatement mappedSt = null;
		if (baseStatementId!= null && configuration.hasStatement(baseStatementId,false)) {
			
			// 复制声明
			MappedStatement baseStatement = configuration.getMappedStatement(baseStatementId,false);
			if (!configuration.hasStatement(statementId,false)) {
				// 创建新的statement,并封装相应的resultType
				mappedSt = cloneStatement(mapperClass, mapperMethod, entityClass, baseStatement, statementId,
						baseStatementId);
			}
		} else {
			// 通过sql builder创建
			SqlBuilder sqlBuilder;
			try {
				sqlBuilder = createSqlBuilder(mapperClass,methodName,entityClass);
			} catch (Exception e) {
				throw new RuntimeException("build statement error[id:" + statementId + "]", e);
			}

			mappedSt = this.buildNewStatement(mapperClass, mapperMethod, entityClass, statementId, sqlBuilder);

		}
		if (mappedSt != null) {
			// 添加进configuration即可
			configuration.addMappedStatement(mappedSt);
		}
		return statementId;
	}
	
	protected SqlBuilder createSqlBuilder(Class<?> mapperMapper,String methodName,Class<?> entityClass) throws InstantiationException, IllegalAccessException, IllegalArgumentException, InvocationTargetException, ClassNotFoundException, NoSuchMethodException, SecurityException {
		// 通过sql builder创建
		SqlBuilder sqlBuilder;
		Class<?> sqlSourceClass = HBatisSqlBuilderFactory.getSqlBuilderClass(this.dialect, methodName);
		Constructor<?> constructor = sqlSourceClass.getDeclaredConstructor(SqlSourceBuilder.class, Class.class);
		sqlBuilder = (SqlBuilder) constructor.newInstance(this.sqlSourceBuilder, entityClass);
		return sqlBuilder;
	}
	/**
	 * 通过SQL BUILDER创建
	 * @param mapperClass
	 * @param mapperMethod
	 * @param entityClass
	 * @param statementId
	 * @param sqlBuilder
	 * @return
	 */
	private MappedStatement buildNewStatement(Class<?> mapperClass, Method mapperMethod, Class<?> entityClass,
			String statementId, SqlBuilder sqlBuilder) {
		MappedStatement.Builder statementBuilder = new MappedStatement.Builder(this.configuration, statementId,
				sqlBuilder, sqlBuilder.getSqlCommandType());
		
		if (SqlCommandType.SELECT.equals(sqlBuilder.getSqlCommandType()) && sqlBuilder.getResultType() != null) {
			List<ResultMap> resultMaps = new ArrayList<ResultMap>();
			ResultMap.Builder resultMapBuilder = new ResultMap.Builder(configuration, statementBuilder.id() + "_Inline",
						sqlBuilder.getResultType(), sqlBuilder.getResultMappingList() == null ? new ArrayList<ResultMapping>(0):sqlBuilder.getResultMappingList());
			resultMaps.add(resultMapBuilder.build());
		
			statementBuilder.resultMaps(resultMaps);
		}
		
		if (SqlCommandType.INSERT.equals(sqlBuilder.getSqlCommandType())) {
			keyGeneratorBuilder.build(statementBuilder, entityClass);
		}

		return statementBuilder.build();
	}
	/**
	 * 通过已声明创建
	 * @param mapperClass
	 * @param method
	 * @param entityClass
	 * @param baseStatement
	 * @param statementId
	 * @param baseStatementId
	 * @return
	 */
	private MappedStatement cloneStatement(Class<?> mapperClass, Method method, Class<?> entityClass,
			MappedStatement baseStatement, String statementId, String baseStatementId) {
		MappedStatement.Builder statementBuilder = new MappedStatement.Builder(baseStatement.getConfiguration(),
				statementId, baseStatement.getSqlSource(), baseStatement.getSqlCommandType());
		statementBuilder.resultMaps(baseStatement.getResultMaps());

		this.setResultMap(statementBuilder, mapperClass, method);

		return statementBuilder.build();
	}
	public static EntityClassDescriber<?> getEntityClassDescriber(Class<?> mapperClass) {
		Class<?> entityClass = getEntityClassByInterface(mapperClass);
		EntityClassDescriber<?> d = EntityClassDescriberHelper.getEntityClassDescriber(entityClass);
		return d;
	}
	private void addBaseColumns(Class<?> mapperClass,EntityClassDescriber<?> mapping) {
		String sid = mapperClass.getName()+"."+HbatisConstants.ENTITY_BASE_COLUMN_LIST_NAME;
		if(!configuration.getSqlFragments().containsKey(sid)) {
			logger.debug("build baseColumns.id:{},baseColumns:{}",sid,mapping.getTableBaseColumns());
			XNode baseColumnNode = XNodeUtil.getRootNode("<script>"+mapping.getTableBaseColumns()+"</script>");
			configuration.getSqlFragments().put(sid, baseColumnNode);
		}
//		String fullid = mapperClass.getName()+"."+HbatisConstants.ENTITY_FULL_COLUMN_LIST_NAME;
//		if(!configuration.getSqlFragments().containsKey(fullid)) {
//			XNode baseColumnNode = XNodeUtil.getRootNode("<script>"+mapping.getTableColumns()+"</script>");
//			configuration.getSqlFragments().put(fullid, baseColumnNode);
//		}
	}
	private void addResultMaps(Class<?> mapperClass,EntityClassDescriber<?> entityDescriber) {
		String resultMapId = mapperClass.getName() +"."+HbatisConstants.ENTITY_RESULTMAP_NAME;
		
		if (!configuration.hasResultMap(resultMapId)) {
			ResultMap rm = ResultMapsBuilder.buildResultMap(resultMapId, configuration, entityDescriber);
			configuration.addResultMap(rm);
			logger.debug("resultMap[id:{}],mappings:\n{}",resultMapId,rm.getResultMappings());
		}
	}
	private void setResultMap(Builder statementBuilder, Class<?> mapperClass, Method m) {
		if (m.getAnnotation(EntityResultMapping.class) != null) {
			String resultMapId = mapperClass.getName() +"."+HbatisConstants.ENTITY_RESULTMAP_NAME;
			statementBuilder.resultMaps(Arrays.asList(configuration.getResultMap(resultMapId)));
		}
	}

	public void setDialect(String dialect) {
		this.dialect = dialect;
	}

}
