package plus.ibatis.hbatis.orm.criteria.support;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.apache.ibatis.exceptions.TooManyResultsException;
import org.springframework.util.Assert;

import plus.ibatis.hbatis.core.AbstractEntityNode;
import plus.ibatis.hbatis.core.EntityNode;
import plus.ibatis.hbatis.core.FieldNode;
import plus.ibatis.hbatis.core.lambda.SFunction;
import plus.ibatis.hbatis.orm.criteria.Orders;
import plus.ibatis.hbatis.orm.criteria.Restrictions;
import plus.ibatis.hbatis.orm.criteria.statement.SelectStatement;
import plus.ibatis.hbatis.orm.mapper.HbatisStatementMapper;

/**
 * EntityQuery
 * @author zz
 * @version 1.0.0
 * @since 1.0.0
 */
public class EntityQuery<T> extends AbstractOrderedEntityQuery<T> {

	private HbatisStatementMapper<T> mapper;

	SelectStatement<T> st;
	public EntityQuery(HbatisStatementMapper<T> mapper, EntityNode<T> n) {
		super(n);
		this.mapper = mapper;
		this.st = StatementBuilder.buildSelect(n);
	}
	/**
	 * select fields
	 * @param field fields
	 * @return EntityQuery
	 */
	@SuppressWarnings({ "unchecked", "rawtypes" })
	public final EntityQuery<T> select(FieldNode ... field) {
		this.st.setFields(Arrays.asList(field));
		return this;
	}
	public final EntityQuery<T> select(List<FieldNode<T,?>> fields) {
		this.st.setFields(fields);
		return this;
	}
	// --- lambda
	/**
	 * select field function
	 * @param func
	 * 			field functions
	 * @return EntityQuery
	 */
	@SafeVarargs
	public final EntityQuery<T> select(SFunction<T, ?> ...func) {
		List<FieldNode<T,?>> selectFields = new ArrayList<>(func.length);
		for(SFunction<T,?> tmp:func ) {
			String prop = prop(tmp);
			FieldNode<T,?> fn = field(prop);
			selectFields.add(fn);
		}
		return this;
	}
	/**
	 * select property 
	 * @param prop property name
	 * @return EntityQuery
	 */
	@SuppressWarnings({ "rawtypes", "unchecked" })
	public final EntityQuery<T> select(String ...prop) {
		List<FieldNode<T,?>> selectFields = new ArrayList<>();
		for(String tmp:prop) {
			FieldNode fn = this.field(tmp);
			selectFields.add(fn);
		}
		this.st.setFields(selectFields);
		return this;
	}
		
	public EntityQuery<T> limit(int limit) {
		this.st.setPageRange(0,limit);
		return this;
	}
	public EntityQuery<T> limit(int start, int limit) {
		this.st.setPageRange(start, limit);
		return this;
	}
	/**
	 * set next page
	 */
	public void nextPage() {
		Assert.notNull(this.st.getPageRange(),"Page params not setted");
		Assert.isTrue(this.st.getPageRange().getLimit() > 0,"Limit of page params should greater then 0");
		this.st.getPageRange().setStart(this.st.getPageRange().getStart() + this.st.getPageRange().getLimit());
	}

	/**
	 * fetch data
	 * 
	 * @return List
	 */
	public List<T> fetch() {
		return this.mapper.selectByStatement(st);
	}
	public T fetchOne() {
		List<T> tmp = this.mapper.selectByStatement(st);
		if (tmp == null || tmp.isEmpty()) {
			return null;
		}
		if(tmp.size()>1) {
			throw new TooManyResultsException("Expect fetch 1,but fetch "+tmp.size());
		}
		return tmp.get(0);
	}
	public T fetchFirst() {
		List<T> tmp = this.mapper.selectByStatement(st);
		if (tmp == null || tmp.isEmpty()) {
			return null;
		}
		return tmp.get(0);
	}
	/**
	 * count
	 * @return count
	 */
	public long count() {
		return this.mapper.countByRestrictions(this.st.restrictions());
	}

	public Restrictions<T> getRestrictions() {
		return st.restrictions();
	}

	@Override
	public Orders<T> getOrders() {
		return this.st.orderby();
	}
	@SuppressWarnings({ "rawtypes", "unchecked" })
	public static <T> EntityQuery<T> create(HbatisStatementMapper<T> repo, EntityNode<T> n) {
		return new EntityQuery(repo, n);
	}
	public static <T> EntityQuery<T> create(HbatisStatementMapper<T> repo,Class<T> clazz){
		return create(repo,AbstractEntityNode.of(clazz));
	}
	public Orders<T> orderBy(Orders<T> orders) {
		this.st.orderby(orders);
		return this.st.orderby();
	}
	public Orders<T> orderBy() {
		return this.st.orderby();
	}
}
