package plus.ibatis.hbatis.plugins;

import java.io.InputStream;
import java.sql.Connection;
import java.sql.SQLException;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Unmarshaller;

import org.apache.ibatis.executor.Executor;
import org.apache.ibatis.logging.Log;
import org.apache.ibatis.logging.LogFactory;
import org.apache.ibatis.mapping.BoundSql;
import org.apache.ibatis.mapping.MappedStatement;
import org.apache.ibatis.mapping.ParameterMapping;
import org.apache.ibatis.mapping.SqlSource;

public class PluginUtils {
	private static final Log logger = LogFactory.getLog(PluginUtils.class);
	private static final Map<String, Dialect> JDBC_DB_TYPE_CACHE = new ConcurrentHashMap<>();

	/**
	 * 不关闭 Connection,因为是从事务里获取的,sqlSession会负责关闭
	 *
	 * @param executor Executor
	 * @return Dialect 
	 */
	public static Dialect getDialect(Executor executor) {
		try {
			Connection conn = executor.getTransaction().getConnection();

			return JDBC_DB_TYPE_CACHE.computeIfAbsent(conn.getMetaData().getURL(), k -> getDialect(k));
		} catch (SQLException e) {
			throw new RuntimeException(e);
		}
	}

	/**
	 * 根据连接地址判断数据库类型
	 *
	 * @param jdbcUrl 连接地址
	 * @return ignore
	 */
	public static Dialect getDialect(String jdbcUrl) {
		String url = jdbcUrl.toLowerCase();
		if (url.contains(":mysql:") || url.contains(":cobar:")) {
			return Dialect.mysql;
		} else if (url.contains(":mariadb:")) {
			return Dialect.mariadb;
		} else if (url.contains(":oracle:")) {
			return Dialect.oracle;
		} else if (url.contains(":sqlserver:") || url.contains(":microsoft:")) {
			return Dialect.sqlserver;
		} else if (url.contains(":sqlserver2012:")) {
			return Dialect.sqlserver2012;
		} else if (url.contains(":postgresql:")) {
			return Dialect.postgresql;
		} else if (url.contains(":hsqldb:")) {
			return Dialect.hsqldb;
		} else if (url.contains(":db2:")) {
			return Dialect.db2;
		} else if (url.contains(":sqlite:")) {
			return Dialect.sqlite;
		} else if (url.contains(":h2:")) {
			return Dialect.h2;
		} else {
			logger.warn("The jdbcUrl is " + jdbcUrl + ", Mybatis Plus Cannot Read Database type or The Database's Not Supported!");
			return Dialect.other;
		}
	}

	public static MappedStatement newMappedStatement(MappedStatement mappedStatement, BoundSql boundSql, String sql) {
		// 重新new一个查询语句对像
		BoundSql newBoundSql = new BoundSql(mappedStatement.getConfiguration(), sql, boundSql.getParameterMappings(), boundSql.getParameterObject());
		// 把新的查询放到statement里
		MappedStatement newMs = copyFromMappedStatement(mappedStatement, new BoundSqlSqlSource(newBoundSql));
		for (ParameterMapping mapping : boundSql.getParameterMappings()) {
			String prop = mapping.getProperty();
			if (boundSql.hasAdditionalParameter(prop)) {
				newBoundSql.setAdditionalParameter(prop, boundSql.getAdditionalParameter(prop));
			}
		}
		return newMs;
	}

	private static MappedStatement copyFromMappedStatement(MappedStatement ms, SqlSource newSqlSource) {
		MappedStatement.Builder builder = new MappedStatement.Builder(ms.getConfiguration(), ms.getId(), newSqlSource, ms.getSqlCommandType());
		builder.resource(ms.getResource());
		builder.fetchSize(ms.getFetchSize());
		builder.statementType(ms.getStatementType());
		builder.keyGenerator(ms.getKeyGenerator());
		if (ms.getKeyProperties() != null && ms.getKeyProperties().length > 0) {
			builder.keyProperty(ms.getKeyProperties()[0]);
		}
		builder.timeout(ms.getTimeout());
		builder.parameterMap(ms.getParameterMap());
		builder.resultMaps(ms.getResultMaps());
		builder.resultSetType(ms.getResultSetType());
		builder.cache(ms.getCache());
		builder.flushCacheRequired(ms.isFlushCacheRequired());
		builder.useCache(ms.isUseCache());
		return builder.build();
	}

	public static class BoundSqlSqlSource implements SqlSource {
		private BoundSql boundSql;

		public BoundSqlSqlSource(BoundSql boundSql) {
			this.boundSql = boundSql;
		}

		public BoundSql getBoundSql(Object parameterObject) {
			return boundSql;
		}
	}
	
	/**
	 * read xml
	 * @param <T> pojo class
	 * @param is inputstream
	 * @param clazz
	 * 			pojo class
	 * @return pojo
	 * @throws JAXBException jaxb exception
	 */
	@SuppressWarnings("unchecked")
	public static <T> T readXml(InputStream is,Class<T> clazz) throws JAXBException {
		T t = null;
		JAXBContext context = JAXBContext.newInstance(clazz);
		Unmarshaller unmarshaller = context.createUnmarshaller();
		t = (T) unmarshaller.unmarshal(is);
		return t;
	}
}
