/*
 * Copyright (C) 2014 Dmitry Kotlyarov, Dmitriy Rogozhin.
 * All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package apphub;

import apphub.config.Transformer;
import apphub.config.transformer.IntTransformer;
import apphub.storage.StorageType;
import apphub.util.PropertyUtil;
import org.jboss.resteasy.client.jaxrs.ResteasyClientBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.Serializable;
import java.net.URLStreamHandler;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;

/**
 * @author Dmitry Kotlyarov
 * @since 1.0
 */
public class Config {
    private static final Logger log = LoggerFactory.getLogger(Config.class);
    private static final Config config = createConfig();

    protected final String domain = PropertyUtil.getSystemProperty("APPHUB_DOMAIN");
    protected final String environment = PropertyUtil.getSystemProperty("APPHUB_ENVIRONMENT");
    protected final EnvironmentType environmentType = EnvironmentType.valueOf(PropertyUtil.getSystemProperty("APPHUB_ENVIRONMENT_TYPE"));
    protected final String environmentDomain = PropertyUtil.getSystemProperty("APPHUB_ENVIRONMENT_DOMAIN");
    protected final StorageType storageType = StorageType.valueOf(PropertyUtil.getSystemProperty("APPHUB_STORAGE_TYPE"));
    protected final String storageProtocol = PropertyUtil.getSystemProperty("APPHUB_STORAGE_PROTOCOL");
    protected final String storageUrl = PropertyUtil.getSystemProperty("APPHUB_STORAGE_URL");
    protected final String serviceDomain = PropertyUtil.getSystemProperty("APPHUB_SERVICE_DOMAIN");
    protected final String serviceUrl = PropertyUtil.getSystemProperty("APPHUB_SERVICE_URL");
    protected final String cdnDomain = PropertyUtil.getSystemProperty("APPHUB_CDN_DOMAIN");
    protected final String cdnUrl = PropertyUtil.getSystemProperty("APPHUB_CDN_URL");
    protected final String jenkinsDomain = PropertyUtil.getSystemProperty("APPHUB_JENKINS_DOMAIN");
    protected final String jenkinsUrl = PropertyUtil.getSystemProperty("APPHUB_JENKINS_URL");
    protected final String providerDomain = PropertyUtil.getSystemProperty("APPHUB_PROVIDER_DOMAIN");
    protected final String instance = PropertyUtil.getSystemProperty("APPHUB_INSTANCE");
    protected final String instanceType = PropertyUtil.getSystemProperty("APPHUB_INSTANCE_TYPE");
    protected final String image = PropertyUtil.getSystemProperty("APPHUB_IMAGE");
    protected final String imageLaunch = PropertyUtil.getSystemProperty("APPHUB_IMAGE_LAUNCH");
    protected final String imageManifest = PropertyUtil.getSystemProperty("APPHUB_IMAGE_MANIFEST");
    protected final String zone = PropertyUtil.getSystemProperty("APPHUB_ZONE");
    protected final String profile = PropertyUtil.getSystemProperty("APPHUB_PROFILE");
    protected final String reservation = PropertyUtil.getSystemProperty("APPHUB_RESERVATION");
    protected final String network = PropertyUtil.getSystemProperty("APPHUB_NETWORK");
    protected final String networkDevice = PropertyUtil.getSystemProperty("APPHUB_NETWORK_DEVICE");
    protected final String networkInterface = PropertyUtil.getSystemProperty("APPHUB_NETWORK_INTERFACE");
    protected final Set<String> publicIps = Collections.unmodifiableSet(new LinkedHashSet<>(Arrays.asList(PropertyUtil.getSystemProperty("APPHUB_PUBLIC_IPS").split(" "))));
    protected final Set<String> privateIps = Collections.unmodifiableSet(new LinkedHashSet<>(Arrays.asList(PropertyUtil.getSystemProperty("APPHUB_PRIVATE_IPS").split(" "))));
    protected final String publicHostname = PropertyUtil.getSystemProperty("APPHUB_PUBLIC_HOSTNAME");
    protected final String privateHostname = PropertyUtil.getSystemProperty("APPHUB_PRIVATE_HOSTNAME");
    protected final String owner = PropertyUtil.getSystemProperty("APPHUB_OWNER");
    protected final String vpc = PropertyUtil.getSystemProperty("APPHUB_VPC");
    protected final String vpcCidr = PropertyUtil.getSystemProperty("APPHUB_VPC_CIDR");
    protected final String subnet = PropertyUtil.getSystemProperty("APPHUB_SUBNET");
    protected final String subnetCidr = PropertyUtil.getSystemProperty("APPHUB_SUBNET_CIDR");
    protected final Set<String> securityGroups = Collections.unmodifiableSet(new LinkedHashSet<>(Arrays.asList(PropertyUtil.getSystemProperty("APPHUB_SECURITY_GROUPS").split(" "))));
    protected final String application;
    protected final String version;
    protected final Map<String, Transformer> transformers = createTransformers();
    protected final Map<String, URLStreamHandler> protocolHandlers = createProtocolHandlers();
    protected final ResteasyClientBuilder clientBuilder = new ResteasyClientBuilder();
    protected final AtomicBoolean shutdown = new AtomicBoolean(false);

    protected Config() {
        this.application = "apphub-dev";
        this.version = "1.0-SNAPSHOT";
    }

    public String getDomain() {
        return domain;
    }

    public String getEnvironment() {
        return environment;
    }

    public EnvironmentType getEnvironmentType() {
        return environmentType;
    }

    public String getEnvironmentDomain() {
        return environmentDomain;
    }

    public StorageType getStorageType() {
        return storageType;
    }

    public String getStorageProtocol() {
        return storageProtocol;
    }

    public String getStorageUrl() {
        return storageUrl;
    }

    public String getServiceDomain() {
        return serviceDomain;
    }

    public String getServiceUrl() {
        return serviceUrl;
    }

    public String getCdnDomain() {
        return cdnDomain;
    }

    public String getCdnUrl() {
        return cdnUrl;
    }

    public String getJenkinsDomain() {
        return jenkinsDomain;
    }

    public String getJenkinsUrl() {
        return jenkinsUrl;
    }

    public String getProviderDomain() {
        return providerDomain;
    }

    public String getInstance() {
        return instance;
    }

    public String getInstanceType() {
        return instanceType;
    }

    public String getImage() {
        return image;
    }

    public String getImageLaunch() {
        return imageLaunch;
    }

    public String getImageManifest() {
        return imageManifest;
    }

    public String getZone() {
        return zone;
    }

    public String getProfile() {
        return profile;
    }

    public String getReservation() {
        return reservation;
    }

    public String getNetwork() {
        return network;
    }

    public String getNetworkDevice() {
        return networkDevice;
    }

    public String getNetworkInterface() {
        return networkInterface;
    }

    public Set<String> getPublicIps() {
        return publicIps;
    }

    public Set<String> getPrivateIps() {
        return privateIps;
    }

    public String getPublicHostname() {
        return publicHostname;
    }

    public String getPrivateHostname() {
        return privateHostname;
    }

    public String getOwner() {
        return owner;
    }

    public String getVpc() {
        return vpc;
    }

    public String getVpcCidr() {
        return vpcCidr;
    }

    public String getSubnet() {
        return subnet;
    }

    public String getSubnetCidr() {
        return subnetCidr;
    }

    public Set<String> getSecurityGroups() {
        return securityGroups;
    }

    public String getApplication() {
        return application;
    }

    public String getVersion() {
        return version;
    }

    public Map<String, Transformer> getTransformers() {
        return transformers;
    }

    public Map<String, URLStreamHandler> getProtocolHandlers() {
        return protocolHandlers;
    }

    public ResteasyClientBuilder getClientBuilder() {
        return clientBuilder;
    }

    public boolean isShutdown() {
        return shutdown.get();
    }

    public void shutdown() {
        shutdown.compareAndSet(false, true);
    }

    public static Config get() {
        return config;
    }

    private static Config createConfig() {
        try {
            Class clazz = Class.forName("apphub.CustomConfig");
            return (Config) clazz.newInstance();
        } catch (ClassNotFoundException e) {
            return new Config();
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    private static Map<String, Transformer> createTransformers() {
        LinkedHashMap<String, Transformer> transformers = new LinkedHashMap<>(1);
        Transformer t = new IntTransformer();
        transformers.put(t.getType(), t);
        return Collections.unmodifiableMap(transformers);
    }

    private static Map<String, URLStreamHandler> createProtocolHandlers() {
        LinkedHashMap<String, URLStreamHandler> protocolHandlers = new LinkedHashMap<>(8);
        protocolHandlers.put("app", new apphub.protocol.app.Handler());
        protocolHandlers.put("as", new apphub.protocol.as.Handler());
        protocolHandlers.put("cp", new apphub.protocol.cp.Handler());
        protocolHandlers.put("env", new apphub.protocol.env.Handler());
        protocolHandlers.put("gs", new apphub.protocol.gs.Handler());
        protocolHandlers.put("inst", new apphub.protocol.inst.Handler());
        protocolHandlers.put("s3", new apphub.protocol.s3.Handler());
        protocolHandlers.put("st", new apphub.protocol.st.Handler());
        return Collections.unmodifiableMap(protocolHandlers);
    }
}
