/*
 * Copyright (C) 2014 Dmitry Kotlyarov, Dmitriy Rogozhin.
 * All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package apphub;

import apphub.storage.Storage;
import apphub.storage.StorageFactory;
import apphub.storage.aws.AWSStorageFactory;

import java.util.HashMap;
import java.util.concurrent.atomic.AtomicReference;

/**
 * @author Dmitry Kotlyarov
 * @since 1.0
 */
public final class Environment {
    public static final StorageFactory STORAGE_FACTORY = createStorageFactory();

    private static final AtomicReference<Environment> environment = new AtomicReference<>(null);
    private static final ThreadLocal<Environment> threadEnvironment = new ThreadLocal<>();

    private final Instance instance = null;
    private final Instance[] instances = new Instance[1];
    private final HashMap<String, Instance> instancesMap = new HashMap<>();
    private final Storage storage = createStorage(Config.get().getEnvironment(), "");

    public static Environment get() {
        Environment environment = threadEnvironment.get();
        return (environment != null) ? environment : Environment.environment.get();
    }

    public static Environment acquire() {
        Environment environment = Environment.environment.get();
        threadEnvironment.set(environment);
        return environment;
    }

    public static Environment release() {
        Environment environment = threadEnvironment.get();
        threadEnvironment.remove();
        return environment;
    }

    public static Storage createStorage(String bucket, String prefix) {
        return STORAGE_FACTORY.create(bucket, prefix);
    }

    private static StorageFactory createStorageFactory() {
        EnvironmentType type = Config.get().getEnvironmentType();
        if (type.equals(EnvironmentType.AWS)) {
            return new AWSStorageFactory();
        } else if (type.equals(EnvironmentType.GCP)) {
            return new AWSStorageFactory();
        } else if (type.equals(EnvironmentType.AZURE)) {
            return new AWSStorageFactory();
        } else if (type.equals(EnvironmentType.OS)) {
            return new AWSStorageFactory();
        } else {
            throw new RuntimeException(String.format("Illegal environment type '%s'", type));
        }
    }
}
