package pro.felixo.logicalclocks.hybrid

import kotlinx.coroutines.sync.Mutex
import kotlinx.coroutines.sync.withLock
import pro.felixo.logicalclocks.LogicalClock

/**
 * A hybrid logical clock (HLC) that combines a physical clock with a logical component.
 *
 * It is much like a Lamport clock in that can be used to provide a causal ordering of events. Its advantage over
 * a Lamport clock is that it can additionally order concurrent events using physical time. Note that this advantage is
 * however lost in cases where a participant's physical clock is faulty, especially if it returns times significantly
 * into the future.
 *
 * Note that the physical component of a timestamp generated by this clock does not necessarily correspond to the actual
 * physical time when the timestamp was generated.
 *
 * This implementation is thread-safe.
 *
 * Example instantiation where physical time is expressed as an `Instant` and logical time as a `Long`:
 *
 * ```
 * import kotlinx.datetime.Clock
 * import kotlinx.datetime.Instant
 *
 * fun instantLongHybridClock(
 *     initialTime: HybridTimestamp<Instant, Long>,
 *     onNewTime: suspend (HybridTimestamp<Instant, Long>) -> Unit = {},
 *     physicalClock: Clock = Clock.System
 * ): HybridClock<Instant, Long> = HybridClock(
 *     initialTime,
 *     0,
 *     { physicalClock.now() },
 *     Long::inc,
 *     onNewTime
 * )
 * ```
 *
 * Usage:
 *
 * ```
 * // Obtain a new timestamp for an event that occurred locally:
 * val timestamp = clock.tick()
 *
 * // Make the clock aware of an event that occurred externally, so it can adjust its external state:
 * clock.tock(externalTimestamp)
 * ```
 */
class HybridClock<Physical : Comparable<Physical>, Logical : Comparable<Logical>>(
    initialTime: HybridTimestamp<Physical, Logical>,
    private val defaultLogical: Logical,
    private val currentPhysical: () -> Physical,
    private val incrementLogical: (Logical) -> Logical,
    private val onNewTime: suspend (HybridTimestamp<Physical, Logical>) -> Unit = {}
) : LogicalClock<HybridTimestamp<Physical, Logical>> {
    private val mutex = Mutex()

    override var lastTime: HybridTimestamp<Physical, Logical> = initialTime
        private set

    override suspend fun tick(): HybridTimestamp<Physical, Logical> = mutex.withLock {
        val currentPhysical = currentPhysical()
        lastTime.run {
            (
                if (currentPhysical > physical)
                    HybridTimestamp(currentPhysical, defaultLogical)
                else
                    HybridTimestamp(physical, incrementLogical(logical))
            ).also { update(it) }
        }
    }

    override suspend fun tock(
        externalTime: HybridTimestamp<Physical, Logical>
    ): HybridTimestamp<Physical, Logical> = mutex.withLock  {
        lastTime.run {
            if (externalTime > this) {
                externalTime.also { update(it) }
            } else
                this
        }
    }

    private suspend fun update(it: HybridTimestamp<Physical, Logical>) {
        lastTime = it
        onNewTime(it)
    }
}
