/*
 * Copyright (c) 2017.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package pro.leaco.mybatis.generator.plugins;

import org.mybatis.generator.api.IntrospectedColumn;
import org.mybatis.generator.api.IntrospectedTable;
import org.mybatis.generator.api.dom.java.*;
import org.mybatis.generator.internal.util.JavaBeansUtil;
import org.mybatis.generator.internal.util.StringUtility;
import pro.leaco.mybatis.generator.plugins.utils.BasePlugin;
import pro.leaco.mybatis.generator.plugins.utils.FormatTools;
import pro.leaco.mybatis.generator.plugins.utils.JavaElementGeneratorTools;
import pro.leaco.mybatis.generator.plugins.utils.PluginTools;
import pro.leaco.mybatis.generator.plugins.utils.hook.IModelColumnPluginHook;

/**
 * ---------------------------------------------------------------------------
 * 数据Model属性对应Column获取插件
 * ---------------------------------------------------------------------------
 *
 * 
 * 
 * ---------------------------------------------------------------------------
 */
public class ModelColumnPlugin extends BasePlugin {
    /**
     * 内部Enum名
     */
    public static final String ENUM_NAME = "Column";

    /**
     * 自定义方法
     */
    public static final String METHOD_EXCLUDES = "excludes";
    public static final String METHOD_ALL = "all";
    public static final String METHOD_AS = "as";
    public static final String METHOD_GET_ESCAPED_COLUMN_NAME = "getEscapedColumnName";
    public static final String METHOD_GET_ALIASED_ESCAPED_COLUMN_NAME = "getAliasedEscapedColumnName";

    public static final String CONST_BEGINNING_DELIMITER = "BEGINNING_DELIMITER";   // const
    public static final String CONST_ENDING_DELIMITER = "ENDING_DELIMITER";   // const

    /**
     * Model Methods 生成
     * 具体执行顺序 http://www.mybatis.org/generator/reference/pluggingIn.html
     *
     * @param topLevelClass
     * @param introspectedTable
     * @return
     */
    @Override
    public boolean modelBaseRecordClassGenerated(TopLevelClass topLevelClass, IntrospectedTable introspectedTable) {
        //增加表格登记
        Field field = new Field("TABLE", FullyQualifiedJavaType.getStringInstance());
        field.setStatic(true);
        field.setVisibility(JavaVisibility.PUBLIC);
        field.setInitializationString("\"" + introspectedTable.getTableConfiguration().getTableName() + "\"");
        commentGenerator.addFieldComment(field, introspectedTable);
        topLevelClass.addField(field);


        //增加枚举列
        topLevelClass.addInnerEnum(this.generateColumnEnum(topLevelClass, introspectedTable));
        return super.modelBaseRecordClassGenerated(topLevelClass, introspectedTable);
    }

    /**
     * Model Methods 生成
     * 具体执行顺序 http://www.mybatis.org/generator/reference/pluggingIn.html
     *
     * @param topLevelClass
     * @param introspectedTable
     * @return
     */
    @Override
    public boolean modelRecordWithBLOBsClassGenerated(TopLevelClass topLevelClass, IntrospectedTable introspectedTable) {
        topLevelClass.addInnerEnum(this.generateColumnEnum(topLevelClass, introspectedTable));
        return super.modelRecordWithBLOBsClassGenerated(topLevelClass, introspectedTable);
    }

    /**
     * 具体执行顺序 http://www.mybatis.org/generator/reference/pluggingIn.html
     *
     * @param topLevelClass
     * @param introspectedTable
     * @return
     */
    @Override
    public boolean modelPrimaryKeyClassGenerated(TopLevelClass topLevelClass, IntrospectedTable introspectedTable) {
        topLevelClass.addInnerEnum(this.generateColumnEnum(topLevelClass, introspectedTable));
        return super.modelPrimaryKeyClassGenerated(topLevelClass, introspectedTable);
    }

    /**
     * 生成Column字段枚举
     *
     * @param topLevelClass
     * @param introspectedTable
     * @return
     */
    private InnerEnum generateColumnEnum(TopLevelClass topLevelClass, IntrospectedTable introspectedTable) {

        String TABLE_NAME = introspectedTable.getTableConfiguration().getTableName();

        // 生成内部枚举
        InnerEnum innerEnum = new InnerEnum(new FullyQualifiedJavaType(ENUM_NAME));
        innerEnum.setVisibility(JavaVisibility.PUBLIC);
        innerEnum.setStatic(true);
        commentGenerator.addEnumComment(innerEnum, introspectedTable);
        logger.debug("itfsw(数据Model属性对应Column获取插件):" + topLevelClass.getType().getShortName() + "增加内部Builder类。");

        // 生成常量
        Field beginningDelimiterField = JavaElementGeneratorTools.generateField(
                CONST_BEGINNING_DELIMITER,
                JavaVisibility.PRIVATE,
                FullyQualifiedJavaType.getStringInstance(),
                "\"" + StringUtility.escapeStringForJava(context.getBeginningDelimiter()) + "\""
        );
        beginningDelimiterField.setStatic(true);
        beginningDelimiterField.setFinal(true);
        commentGenerator.addFieldComment(beginningDelimiterField, introspectedTable);
        innerEnum.addField(beginningDelimiterField);

        Field endingDelimiterField = JavaElementGeneratorTools.generateField(
                CONST_ENDING_DELIMITER,
                JavaVisibility.PRIVATE,
                FullyQualifiedJavaType.getStringInstance(),
                "\"" + StringUtility.escapeStringForJava(context.getEndingDelimiter()) + "\""
        );
        endingDelimiterField.setStatic(true);
        endingDelimiterField.setFinal(true);
        commentGenerator.addFieldComment(endingDelimiterField, introspectedTable);
        innerEnum.addField(endingDelimiterField);

        // 生成属性和构造函数
        Field columnField = new Field("column", FullyQualifiedJavaType.getStringInstance());
        columnField.setVisibility(JavaVisibility.PRIVATE);
        columnField.setFinal(true);
        commentGenerator.addFieldComment(columnField, introspectedTable);
        innerEnum.addField(columnField);

        Field isColumnNameDelimitedField = new Field("isColumnNameDelimited", FullyQualifiedJavaType.getBooleanPrimitiveInstance());
        isColumnNameDelimitedField.setVisibility(JavaVisibility.PRIVATE);
        isColumnNameDelimitedField.setFinal(true);
        commentGenerator.addFieldComment(isColumnNameDelimitedField, introspectedTable);
        innerEnum.addField(isColumnNameDelimitedField);

        Field javaPropertyField = new Field("javaProperty", FullyQualifiedJavaType.getStringInstance());
        javaPropertyField.setVisibility(JavaVisibility.PRIVATE);
        javaPropertyField.setFinal(true);
        commentGenerator.addFieldComment(javaPropertyField, introspectedTable);
        innerEnum.addField(javaPropertyField);

        Field jdbcTypeField = new Field("jdbcType", FullyQualifiedJavaType.getStringInstance());
        jdbcTypeField.setVisibility(JavaVisibility.PRIVATE);
        jdbcTypeField.setFinal(true);
        commentGenerator.addFieldComment(jdbcTypeField, introspectedTable);
        innerEnum.addField(jdbcTypeField);

        Method mValue = new Method("value");
        mValue.setVisibility(JavaVisibility.PUBLIC);
        mValue.setReturnType(FullyQualifiedJavaType.getStringInstance());
        mValue.addBodyLine("return this.column;");
        commentGenerator.addGeneralMethodComment(mValue, introspectedTable);
        FormatTools.addMethodWithBestPosition(innerEnum, mValue);

        Method mGetValue = new Method("getValue");
        mGetValue.setVisibility(JavaVisibility.PUBLIC);
        mGetValue.setReturnType(FullyQualifiedJavaType.getStringInstance());
        mGetValue.addBodyLine("return this.column;");
        commentGenerator.addGeneralMethodComment(mGetValue, introspectedTable);
        FormatTools.addMethodWithBestPosition(innerEnum, mGetValue);

        Method mGetJavaProperty = JavaElementGeneratorTools.generateGetterMethod(javaPropertyField);
        commentGenerator.addGeneralMethodComment(mGetJavaProperty, introspectedTable);
        FormatTools.addMethodWithBestPosition(innerEnum, mGetJavaProperty);

        Method mGetJdbcType = JavaElementGeneratorTools.generateGetterMethod(jdbcTypeField);
        commentGenerator.addGeneralMethodComment(mGetJdbcType, introspectedTable);
        FormatTools.addMethodWithBestPosition(innerEnum, mGetJdbcType);

        Method constructor = new Method(ENUM_NAME);
        constructor.setConstructor(true);
        constructor.addBodyLine("this.column = column;");
        constructor.addBodyLine("this.javaProperty = javaProperty;");
        constructor.addBodyLine("this.jdbcType = jdbcType;");
        constructor.addBodyLine("this.isColumnNameDelimited = isColumnNameDelimited;");
        constructor.addParameter(new Parameter(FullyQualifiedJavaType.getStringInstance(), "column"));
        constructor.addParameter(new Parameter(FullyQualifiedJavaType.getStringInstance(), "javaProperty"));
        constructor.addParameter(new Parameter(FullyQualifiedJavaType.getStringInstance(), "jdbcType"));
        constructor.addParameter(new Parameter(FullyQualifiedJavaType.getBooleanPrimitiveInstance(), "isColumnNameDelimited"));
        commentGenerator.addGeneralMethodComment(constructor, introspectedTable);
        FormatTools.addMethodWithBestPosition(innerEnum, constructor);
        logger.debug("itfsw(数据Model属性对应Column获取插件):" + topLevelClass.getType().getShortName() + ".Column增加构造方法和column属性。");

        // Enum枚举
        for (IntrospectedColumn introspectedColumn : introspectedTable.getAllColumns()) {
            Field field = JavaBeansUtil.getJavaBeansField(introspectedColumn, context, introspectedTable);

            StringBuffer sb = new StringBuffer();
            sb.append(field.getName());
            sb.append("(\"");
            sb.append(introspectedColumn.getActualColumnName());
            sb.append("\", \"");
            sb.append(introspectedColumn.getJavaProperty());
            sb.append("\", \"");
            sb.append(introspectedColumn.getJdbcTypeName());
            sb.append("\", ");
            sb.append(introspectedColumn.isColumnNameDelimited());
            sb.append(")");

            innerEnum.addEnumConstant(sb.toString());
            logger.debug("itfsw(数据Model属性对应Column获取插件):" + topLevelClass.getType().getShortName() + ".Column增加" + field.getName() + "枚举。");
        }

        // asc 和 desc 方法
        Method desc = new Method("desc");
        desc.setVisibility(JavaVisibility.PUBLIC);
        desc.setReturnType(FullyQualifiedJavaType.getStringInstance());
        desc.addBodyLine("return this." + METHOD_GET_ESCAPED_COLUMN_NAME + "() + \" DESC\";");
        commentGenerator.addGeneralMethodComment(desc, introspectedTable);
        FormatTools.addMethodWithBestPosition(innerEnum, desc);

        Method asc = new Method("asc");
        asc.setVisibility(JavaVisibility.PUBLIC);
        asc.setReturnType(FullyQualifiedJavaType.getStringInstance());
        asc.addBodyLine("return this." + METHOD_GET_ESCAPED_COLUMN_NAME + "() + \" ASC\";");
        commentGenerator.addGeneralMethodComment(asc, introspectedTable);
        FormatTools.addMethodWithBestPosition(innerEnum, asc);
        logger.debug("itfsw(数据Model属性对应Column获取插件):" + topLevelClass.getType().getShortName() + ".Column增加asc()和desc()方法。");

        // as
        Method mAs = new Method(METHOD_AS);
        mAs.setVisibility(JavaVisibility.PUBLIC);
        mAs.setReturnType(FullyQualifiedJavaType.getStringInstance());
        mAs.addBodyLine("return this." + METHOD_GET_ESCAPED_COLUMN_NAME + "() + \" AS \" + this.javaProperty;");
        commentGenerator.addGeneralMethodComment(mAs, introspectedTable);
        FormatTools.addMethodWithBestPosition(innerEnum, mAs);
        Method mAs1 = new Method(METHOD_AS);
        mAs1.setVisibility(JavaVisibility.PUBLIC);
        mAs1.addParameter(new Parameter(FullyQualifiedJavaType.getStringInstance(), "alias"));
        mAs1.setReturnType(FullyQualifiedJavaType.getStringInstance());
        mAs1.addBodyLine("return this." + METHOD_GET_ESCAPED_COLUMN_NAME + "() + \" AS \" + alias;");
        commentGenerator.addGeneralMethodComment(mAs1, introspectedTable);
        FormatTools.addMethodWithBestPosition(innerEnum, mAs1);
        logger.debug("itfsw(数据Model属性对应Column获取插件):" + topLevelClass.getType().getShortName() + ".Column增加as(string)方法。");

        //增加快速生成mysql语句的方法
        addMysqlFunctionWrapper("count", "COUNT", null, topLevelClass, introspectedTable, innerEnum);
        addMysqlFunctionWrapper("sum", "SUM", null, topLevelClass, introspectedTable, innerEnum);
        addMysqlFunctionWrapper("year", "YEAR", null, topLevelClass, introspectedTable, innerEnum);
        addMysqlFunctionWrapper("firstDayOfYear", "DATE_FORMAT(DATE_SUB", "INTERVAL DAYOFYEAR(" + "\" + this." + METHOD_GET_ESCAPED_COLUMN_NAME + "() + \"" + ")-1 DAY) ,'%Y-%m-%d'", topLevelClass, introspectedTable, innerEnum);
        addMysqlFunctionWrapper("quarter", "QUARTER", null, topLevelClass, introspectedTable, innerEnum);
        addMysqlFunctionWrapper("month", "MONTH", null, topLevelClass, introspectedTable, innerEnum);
        addMysqlFunctionWrapper("firstDayOfMonth", "DATE_FORMAT(DATE_SUB", "INTERVAL DAY(" + "\" + this." + METHOD_GET_ESCAPED_COLUMN_NAME + "() + \"" + ")-1 DAY) ,'%Y-%m-%d'", topLevelClass, introspectedTable, innerEnum);
        //周一优先的一周（例： 2020092）
        addMysqlFunctionWrapper("yearWeekMondayFirst", "YEARWEEK(date_format", "'%Y-%m-%d'),1", topLevelClass, introspectedTable, innerEnum);
        addMysqlFunctionWrapper("firstDayOfWeek", "DATE_FORMAT(DATE_ADD", "INTERVAL -WEEKDAY(" + "\" + this." + METHOD_GET_ESCAPED_COLUMN_NAME + "() + \"" + ") DAY) ,'%Y-%m-%d'", topLevelClass, introspectedTable, innerEnum);
        addMysqlFunctionWrapper("day", "DAY", null, topLevelClass, introspectedTable, innerEnum);
        addMysqlFunctionWrapper("toDays", "TO_DAYS", null, topLevelClass, introspectedTable, innerEnum);
        addMysqlFunctionWrapper("dateFormatWith$yyyyMMdd$", "DATE_FORMAT", "\\\"%Y-%m-%d\\\"", topLevelClass, introspectedTable, innerEnum);
        addMysqlFunctionWrapper("dateFormatWith$yyyyMMddHHmmss$", "DATE_FORMAT", "\\\"%Y-%m-%d %T\\\"", topLevelClass, introspectedTable, innerEnum);
        addMysqlFunctionWrapper("dateFormat", "DATE_FORMAT", "pattern", true, topLevelClass, introspectedTable, innerEnum);
        addMysqlFunctionWrapper("dateSub", "DATE_SUB", "param", false, topLevelClass, introspectedTable, innerEnum);


        // excludes
        topLevelClass.addImportedType("java.util.Arrays");
        topLevelClass.addImportedType(FullyQualifiedJavaType.getNewArrayListInstance());
        Method mExcludes = JavaElementGeneratorTools.generateMethod(
                METHOD_EXCLUDES,
                false, JavaVisibility.PUBLIC,
                new FullyQualifiedJavaType(ENUM_NAME + "[]"),
                new Parameter(innerEnum.getType(), "excludes", true)
        );
        commentGenerator.addGeneralMethodComment(mExcludes, introspectedTable);
        mExcludes.setStatic(true);
        JavaElementGeneratorTools.generateMethodBody(
                mExcludes,
                "ArrayList<Column> columns = new ArrayList<>(Arrays.asList(Column.values()));",
                "if (excludes != null && excludes.length > 0) {",
                "columns.removeAll(new ArrayList<>(Arrays.asList(excludes)));",
                "}",
                "return columns.toArray(new Column[]{});"
        );
        FormatTools.addMethodWithBestPosition(innerEnum, mExcludes);
        logger.debug("itfsw(数据Model属性对应Column获取插件):" + topLevelClass.getType().getShortName() + ".Column增加excludes方法。");

        // all
        Method mAll = JavaElementGeneratorTools.generateMethod(
                METHOD_ALL,
                false, JavaVisibility.PUBLIC,
                new FullyQualifiedJavaType(ENUM_NAME + "[]")
        );
        commentGenerator.addGeneralMethodComment(mAll, introspectedTable);
        mAll.setStatic(true);
        mAll.addBodyLine("return Column.values();");
        FormatTools.addMethodWithBestPosition(innerEnum, mAll);
        logger.debug("itfsw(数据Model属性对应Column获取插件):" + topLevelClass.getType().getShortName() + ".Column增加all方法。");


        // getEscapedColumnName
        Method mGetAliasame = JavaElementGeneratorTools.generateMethod(
                "withAlias",
                false, JavaVisibility.PUBLIC,
                FullyQualifiedJavaType.getStringInstance()
        );
        commentGenerator.addGeneralMethodComment(mGetAliasame, introspectedTable);
        JavaElementGeneratorTools.generateMethodBody(
                mGetAliasame,
                "if (this.isColumnNameDelimited) {",
                "return \"" + TABLE_NAME + ".\" + " + CONST_BEGINNING_DELIMITER + " + this.column + " + CONST_ENDING_DELIMITER + ";",
                "} else {",
                "return \"" + TABLE_NAME + ".\" + this.column;",
                "}"
        );
        FormatTools.addMethodWithBestPosition(innerEnum, mGetAliasame);

        // getEscapedColumnName
        Method mGetAliasAs = JavaElementGeneratorTools.generateMethod(
                "withAlias",
                false, JavaVisibility.PUBLIC,
                FullyQualifiedJavaType.getStringInstance(),
                new Parameter(FullyQualifiedJavaType.getStringInstance(), "as")
        );
        commentGenerator.addGeneralMethodComment(mGetAliasAs, introspectedTable);
        JavaElementGeneratorTools.generateMethodBody(
                mGetAliasAs,
                "if (this.isColumnNameDelimited) {",
                "return \"" + TABLE_NAME + ".\" + " + CONST_BEGINNING_DELIMITER + " + this.column + " + CONST_ENDING_DELIMITER + " + \" as \" + as;",
                "} else {",
                "return \"" + TABLE_NAME + ".\" + this.column + \" as \" + as;",
                "}"
        );
        FormatTools.addMethodWithBestPosition(innerEnum, mGetAliasAs);
        logger.debug("itfsw(数据Model属性对应Column获取插件):" + topLevelClass.getType().getShortName() + ".Column增加 alias 方法。");


        // getEscapedColumnName
        Method mGetEscapedColumnName = JavaElementGeneratorTools.generateMethod(
                METHOD_GET_ESCAPED_COLUMN_NAME,
                false, JavaVisibility.PUBLIC,
                FullyQualifiedJavaType.getStringInstance()
        );
        commentGenerator.addGeneralMethodComment(mGetEscapedColumnName, introspectedTable);
        JavaElementGeneratorTools.generateMethodBody(
                mGetEscapedColumnName,
                "if (this.isColumnNameDelimited) {",
                "return " + CONST_BEGINNING_DELIMITER + " + this.column + " + CONST_ENDING_DELIMITER + ";",
                "} else {",
                "return this.column;",
                "}"
        );
        FormatTools.addMethodWithBestPosition(innerEnum, mGetEscapedColumnName);
        logger.debug("itfsw(数据Model属性对应Column获取插件):" + topLevelClass.getType().getShortName() + ".Column增加getEscapedColumnName方法。");

        // getAliasedEscapedColumnName
        Method mGetAliasedEscapedColumnName = JavaElementGeneratorTools.generateMethod(
                METHOD_GET_ALIASED_ESCAPED_COLUMN_NAME,
                false, JavaVisibility.PUBLIC,
                FullyQualifiedJavaType.getStringInstance()
        );
        commentGenerator.addGeneralMethodComment(mGetAliasedEscapedColumnName, introspectedTable);
        if (StringUtility.stringHasValue(introspectedTable.getTableConfiguration().getAlias())) {
            String alias = introspectedTable.getTableConfiguration().getAlias();
            mGetAliasedEscapedColumnName.addBodyLine("StringBuilder sb = new StringBuilder();");
            mGetAliasedEscapedColumnName.addBodyLine("sb.append(\"" + alias + ".\");");
            mGetAliasedEscapedColumnName.addBodyLine("sb.append(this." + METHOD_GET_ESCAPED_COLUMN_NAME + "());");
            mGetAliasedEscapedColumnName.addBodyLine("sb.append(\" as \");");
            mGetAliasedEscapedColumnName.addBodyLine("if (this.isColumnNameDelimited) {");
            mGetAliasedEscapedColumnName.addBodyLine("sb.append(" + CONST_BEGINNING_DELIMITER + ");");
            mGetAliasedEscapedColumnName.addBodyLine("}");
            mGetAliasedEscapedColumnName.addBodyLine("sb.append(\"" + alias + "_\");");
            mGetAliasedEscapedColumnName.addBodyLine("sb.append(this.column);");
            mGetAliasedEscapedColumnName.addBodyLine("if (this.isColumnNameDelimited) {");
            mGetAliasedEscapedColumnName.addBodyLine("sb.append(" + CONST_BEGINNING_DELIMITER + ");");
            mGetAliasedEscapedColumnName.addBodyLine("}");
            mGetAliasedEscapedColumnName.addBodyLine("return sb.toString();");
        } else {
            mGetAliasedEscapedColumnName.addBodyLine("return this." + METHOD_GET_ESCAPED_COLUMN_NAME + "();");
        }
        FormatTools.addMethodWithBestPosition(innerEnum, mGetAliasedEscapedColumnName);
        logger.debug("itfsw(数据Model属性对应Column获取插件):" + topLevelClass.getType().getShortName() + ".Column增加getAliasedEscapedColumnName方法。");

        // hook
        PluginTools.getHook(IModelColumnPluginHook.class).modelColumnEnumGenerated(innerEnum, topLevelClass, introspectedTable);

        return innerEnum;
    }

    /**
     * @param methodName        追加的MYSQL辅助包装函数名
     * @param mysqlFunctionName MYSQL函数名
     * @param topLevelClass
     * @param introspectedTable
     * @param innerEnum
     */
    private void addMysqlFunctionWrapper(
            String methodName,
            String mysqlFunctionName,
            String mysqlFunctionInitParam,
            TopLevelClass topLevelClass,
            IntrospectedTable introspectedTable, InnerEnum innerEnum) {

        String body = "return \"" + mysqlFunctionName + "(" +
                "\" + this." + METHOD_GET_ESCAPED_COLUMN_NAME + "() + \"" +
                (mysqlFunctionInitParam != null ? " , " + mysqlFunctionInitParam : "") +
                ")";

        Method m = new Method(methodName);
        m.setVisibility(JavaVisibility.PUBLIC);
        m.setReturnType(FullyQualifiedJavaType.getStringInstance());
        m.addBodyLine(body + "\";");
        commentGenerator.addGeneralMethodComment(m, introspectedTable);
        FormatTools.addMethodWithBestPosition(innerEnum, m);

        Method m1 = new Method(methodName + "As");
        m1.setVisibility(JavaVisibility.PUBLIC);
        m1.setReturnType(FullyQualifiedJavaType.getStringInstance());
        m1.addBodyLine(body + " AS \" + this.getJavaProperty();");
        commentGenerator.addGeneralMethodComment(m1, introspectedTable);
        FormatTools.addMethodWithBestPosition(innerEnum, m1);

        Method m2 = new Method(methodName + "As");
        m2.setVisibility(JavaVisibility.PUBLIC);
        m2.addParameter(new Parameter(FullyQualifiedJavaType.getStringInstance(), "alias"));
        m2.setReturnType(FullyQualifiedJavaType.getStringInstance());
        m2.addBodyLine(body + " AS \" + alias;");
        commentGenerator.addGeneralMethodComment(m2, introspectedTable);
        FormatTools.addMethodWithBestPosition(innerEnum, m2);
        logger.debug("itfsw(数据Model属性对应Column获取插件):" + topLevelClass.getType().getShortName() + ".Column增加" + mysqlFunctionName + "()方法。");
    }


    /**
     * @param methodName             追加的MYSQL辅助包装函数名
     * @param mysqlFunctionName      MYSQL函数名
     * @param mysqlFunctionParamName 需要叠加的参数名
     * @param withQuot               生成的方法是否需要带引号
     * @param topLevelClass
     * @param introspectedTable
     * @param innerEnum
     */
    private void addMysqlFunctionWrapper(
            String methodName,
            String mysqlFunctionName,
            String mysqlFunctionParamName,
            boolean withQuot,
            TopLevelClass topLevelClass, IntrospectedTable introspectedTable, InnerEnum innerEnum) {
        String body = "return \"" + mysqlFunctionName + "(\" + this." + METHOD_GET_ESCAPED_COLUMN_NAME + "() + \" , ";
        if (withQuot) {
            body += "\\\"";
        }
        body += "\" + " + mysqlFunctionParamName + " + \"";
        if (withQuot) {
            body += "\\\"";
        }
        body += ")";


        Method m = new Method(methodName);
        m.setVisibility(JavaVisibility.PUBLIC);
        m.addParameter(new Parameter(FullyQualifiedJavaType.getStringInstance(), mysqlFunctionParamName));
        m.setReturnType(FullyQualifiedJavaType.getStringInstance());
        m.addBodyLine(body + "\";");
        commentGenerator.addGeneralMethodComment(m, introspectedTable);
        FormatTools.addMethodWithBestPosition(innerEnum, m);

        Method m1 = new Method(methodName + "As");
        m1.setVisibility(JavaVisibility.PUBLIC);
        m1.addParameter(new Parameter(FullyQualifiedJavaType.getStringInstance(), mysqlFunctionParamName));
        m1.setReturnType(FullyQualifiedJavaType.getStringInstance());
        m1.addBodyLine(body + " AS \" + this.getJavaProperty();");
        commentGenerator.addGeneralMethodComment(m1, introspectedTable);
        FormatTools.addMethodWithBestPosition(innerEnum, m1);

        Method m2 = new Method(methodName + "As");
        m2.setVisibility(JavaVisibility.PUBLIC);
        m2.addParameter(new Parameter(FullyQualifiedJavaType.getStringInstance(), mysqlFunctionParamName));
        m2.addParameter(new Parameter(FullyQualifiedJavaType.getStringInstance(), "alias"));
        m2.setReturnType(FullyQualifiedJavaType.getStringInstance());
        m2.addBodyLine(body + " AS \" + alias;");
        commentGenerator.addGeneralMethodComment(m2, introspectedTable);
        FormatTools.addMethodWithBestPosition(innerEnum, m2);


        logger.debug("itfsw(数据Model属性对应Column获取插件):" + topLevelClass.getType().getShortName() + ".Column增加" + mysqlFunctionName + "()方法。");
    }
}
