package pro.mypvit;

import okhttp3.*;
import pro.mypvit.exceptions.ExceptionUtils;
import pro.mypvit.requests.TransactionLinkRequest;
import pro.mypvit.requests.TransactionRestRequest;
import pro.mypvit.responses.*;

import java.io.IOException;

/**
 * MyPVitClient is a client class for interacting with the MyPVit API.
 * It provides methods for initiating transactions, checking status, retrieving KYC information,
 * and other operations with the MyPVit API.
 */
public class MyPVitClient {
    private static final String BASE_URL = "https://api.mypvit.pro";
    private final OkHttpClient client;

    /**
     * Initializes a new instance of MyPVitClient.
     * This constructor sets up the HTTP client used for API requests.
     */
    public MyPVitClient() {
        this.client = new OkHttpClient();
    }

    /**
     * Initializes a REST transaction with the MyPVit API.
     *
     * @param transaction The transaction request data.
     * @param codeUrl     The specific endpoint code URL for the API.
     * @param secretKey   The API secret key for authentication.
     * @return A {@link TransactionRestResponse} containing the API response.
     */
    public TransactionRestResponse initRestTransaction(TransactionRestRequest transaction, String codeUrl, String secretKey) {
        RequestBody body = RequestBody.create(
                transaction.toJson(),
                MediaType.parse("application/json")
        );

        Request request = new Request.Builder()
                .url(BASE_URL.concat("/").concat(codeUrl).concat("/rest"))
                .addHeader("X-Secret", secretKey)
                .addHeader("Accept", "application/json")
                .addHeader("Content-Type", "application/json")
                .post(body)
                .build();

        return sendRequest(request, TransactionRestResponse.class);
    }


    /**
     * Initializes a LINK transaction with the MyPVit API.
     *
     * @param transaction The transaction request data.
     * @param codeUrl     The specific endpoint code URL for the API.
     * @param secretKey   The API secret key for authentication.
     * @return A {@link TransactionLinkResponse} containing the API response.
     */
    public TransactionLinkResponse initLinkTransaction(TransactionLinkRequest transaction, String codeUrl, String secretKey){
        RequestBody body = RequestBody.create(
                transaction.toJson(),
                MediaType.parse("application/json")
        );

        Request request = new Request.Builder()
                .url(BASE_URL.concat("/").concat(codeUrl).concat("/link"))
                .addHeader("X-Secret", secretKey)
                .addHeader("Accept", "application/json")
                .addHeader("Content-Type", "application/json")
                .post(body)
                .build();

        return sendRequest(request, TransactionLinkResponse.class);
    }

    /**
     * Retrieves the status of a specific transaction.
     *
     * @param transactionId         The ID of the transaction to query.
     * @param accountOperationCode  The account operation code.
     * @param transactionOperation  The transaction operation type.
     * @param codeUrl               The specific endpoint code URL for the API.
     * @param secretKey             The API secret key for authentication.
     * @return A {@link StatusResponse} containing the transaction status.
     */
    public StatusResponse getStatus(String transactionId, String accountOperationCode, String transactionOperation, String codeUrl, String secretKey) {
        String uri = "transactionId=".concat(transactionId).concat("&accountOperationCode=").concat(accountOperationCode).concat("&transactionOperation=").concat(transactionOperation);

        Request request = new Request.Builder()
                .url(BASE_URL.concat("/").concat(codeUrl).concat("/status?").concat(uri))
                .addHeader("Accept", "application/json")
                .addHeader("Content-Type", "application/json")
                .addHeader("X-Secret", secretKey)
                .build();

        return sendRequest(request, StatusResponse.class);
    }

    /**
     * Retrieves KYC (Know Your Customer) information for a specific customer.
     *
     * @param customerAccountNumber The customer account number.
     * @param codeUrl               The specific endpoint code URL for the API.
     * @return A {@link KycResponse} containing the KYC information.
     */
    public KycResponse getKyc(String customerAccountNumber, String codeUrl) {
        Request request = new Request.Builder()
                .url(BASE_URL.concat("/").concat(codeUrl).concat("/kyc?customerAccountNumber=").concat(customerAccountNumber))
                .addHeader("Accept", "application/json")
                .addHeader("Content-Type", "application/json")
                .build();

        return sendRequest(request, KycResponse.class);
    }

    /**
     * Retrieves the balance for a specific account.
     *
     * @param accountOperationCode The account operation code.
     * @param codeUrl              The specific endpoint code URL for the API.
     * @param secretKey            The API secret key for authentication.
     * @return A {@link BalanceResponse} containing the account balance.
     */
    public BalanceResponse getBalance(String accountOperationCode, String codeUrl, String secretKey) {
        Request request = new Request.Builder()
                .url(BASE_URL.concat("/").concat(codeUrl).concat("/balance?accountOperationCode=").concat(accountOperationCode))
                .addHeader("Accept", "application/json")
                .addHeader("Content-Type", "application/json")
                .addHeader("X-Secret", secretKey)
                .build();

        return sendRequest(request, BalanceResponse.class);
    }

    /**
     * Retrieves a new token for accessing the MyPVit API.
     *
     * @param accountOperationCode The account operation code.
     * @param receptionUrlCode     The reception URL code.
     * @param password             The account password.
     * @param codeUrl              The specific endpoint code URL for the API.
     * @return A {@link TokenResponse} containing the new token.
     */
    public TokenResponse getToken(String accountOperationCode, String receptionUrlCode, String password, String codeUrl) {
        RequestBody body = new FormBody.Builder()
                .addEncoded("operationAccountCode", accountOperationCode)
                .addEncoded("receptionUrlCode", receptionUrlCode)
                .addEncoded("password", password)
                .build();


        Request request = new Request.Builder()
                .url(BASE_URL.concat("/").concat(codeUrl).concat("/renew-secret"))
                .addHeader("Accept", "application/json")
                .addHeader("Content-Type", "application/json")
                .post(body)
                .build();

        return sendRequest(request, TokenResponse.class);
    }


    /**
     * Sends an HTTP request and parses the response into the specified type.
     *
     * @param request The HTTP request to send.
     * @param clazz   The class of the response type.
     * @param <T>     The type of the response.
     * @return The parsed response.
     * @throws RuntimeException if an I/O error occurs or the response contains an API error.
     */
    private <T extends AbstractResponse> T sendRequest(Request request, Class<T> clazz) {
        try (Response response = client.newCall(request).execute()) {
            ExceptionUtils.handleApiError(response);

            assert response.body() != null;
            return T.fromJson(response.body().string(), clazz);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

}